/// @file types.h
#pragma once

#include <stdint.h>
#include <stddef.h>

#include <enyx/cores_c/symbol_visibility.h>

/**
 * This module contains network related types.
 *
 * These type are used in other modules like @b tcp and @b net_interface.
 */

/**
 * MAC Address type. Bytes are in host byte order.
 *
 * For example with address 11:22:33:44:55:66:
 * @code
 * enyx_mac_address addr = {
 *     .value = { 0x11, 0x22, 0x33, 0x44, 0x55, 0x66 }
 * };
 * @endcode
 */
typedef struct {
    uint8_t value[6];
} enyx_mac_address;

/**
 *  Parse the @p mac address from @p string.
 *
 *  @param string The string to parse
 *  @param mac The address to fill
 *  @return @b 0 on success, @b -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_parse_mac_address(char const * string,
                       enyx_mac_address * mac);
/**
 * Construct a MAC from an @p integer
 *
 * @param integer The integer to construct from.
 * @param mac The address to fill
 * @return @b 0 on success, @b -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_mac_address_from_int(uint64_t integer,
                          enyx_mac_address * mac);

/**
 *  Print the @p mac_address @p buffer of @p buffer_capacity.
 *
 *  @note At most @p buffer_capacity is written to the buffer, even
 *        if the actual string representation is longer.
 *
 *  @param mac_address The address to print
 *  @param buffer The buffer to fill with the string representation
 *  @param buffer_capacity The capacity of the buffer
 *  @return The number of character of the string representation on success,
 *          -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_print_mac_address(enyx_mac_address const* mac_address,
                       char * buffer,
                       size_t buffer_capacity);

/**
 * Convert a @p mac address to an integer type.
 *
 * @param mac The MAC to convert.
 * @return The MAC as an integer.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY uint64_t
enyx_mac_address_to_int(enyx_mac_address const * mac);

/**
 * IPv4 Address type. Bytes are in host byte order.
 *
 * For example with address 192.168.0.1:
 * @code
 * enyx_ipv4 addr = {
 *  .value = { 192, 168, 0, 1 }
 * };
 * @endcode
 */
typedef struct {
    uint8_t value[4];
} enyx_ipv4_address;

/**
 *  Parse the @p ipv4 from @p string.
 *
 *  @param string The string to parse
 *  @param ipv4 The address to fill
 *  @return @b 0 on success, @b -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_parse_ipv4_address(char const * string,
                        enyx_ipv4_address * ipv4);

/**
 * Construct an IPv4 from an @p integer
 *
 * @param integer The integer to construct from.
 * @param ipv4 The address to fill
 * @return @b 0 on success, @b -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ipv4_address_from_int(uint32_t integer,
                           enyx_ipv4_address * ipv4);

/**
 *  Print the @p ipv4 into @p buffer of @p buffer_capacity.
 *
 *  @note At most @p buffer_capacity is written to the buffer, even
 *        if the actual string representation is longer.
 *
 *  @param ipv4 The address to print
 *  @param buffer The buffer to fill with the string representation
 *  @param buffer_capacity The capacity of the buffer
 *  @return The number of character of the string representation on success,
 *          -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_print_ipv4_address(enyx_ipv4_address const* ipv4,
                        char * buffer,
                        size_t buffer_capacity);

/**
 * Convert a @p ipv4 address to an integer type.
 *
 * @param ipv4 The IPv4 address to convert.
 * @return The IPv4 as an integer.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY uint32_t
enyx_ipv4_address_to_int(enyx_ipv4_address const * ipv4);

/**
 * A basic IPv4 endpoint for TCP or UDP protocol.
 */
typedef struct {
    /// Address IPv4 address of endpoint
    enyx_ipv4_address address;
    /// Port of of endpoint.
    uint16_t port;
} enyx_ipv4_endpoint;

/**
 *  Parse the @p endpoint from @p string.
 *
 *  @param string The string to parse
 *  @param endpoint The endpoint to fill
 *  @return @b 0 on success, @b -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_parse_ipv4_endpoint(char const * string,
                         enyx_ipv4_endpoint * endpoint);

/**
 *  Print the @p endpoint into @p buffer of @p buffer_capacity.
 *
 *  @note At most @p buffer_capacity is written to the buffer, even
 *        if the actual string representation is longer.
 *
 *  @param endpoint The endpoint to print
 *  @param buffer The buffer to fill with the string representation
 *  @param buffer_capacity The capacity of the buffer
 *  @return The number of character of the string representation on success,
 *          -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_print_ipv4_endpoint(enyx_ipv4_endpoint const* endpoint,
                         char * buffer,
                         size_t buffer_capacity);
