#pragma once

#include <stdbool.h>
#include <stdint.h>
#include <unistd.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/tcp/session.h>

/**
 * The TCP subsystem of a network stack.
 */
typedef struct enyx_tcp enyx_tcp;

/**
 * Construct the TCP module
 *
 * This module configures the following hw core:
 *
 * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
 * * tcp_multi_stack (from TCP_STD 2.x)
 *
 * @param stack The hardware core containing the stack.
 * @note This requires you to find a compatible enyx_hw_core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_tcp *
enyx_tcp_create(enyx_hw_core * stack);

/**
 * Destroy a TCP subsystem.
 *
 * @param tcp The subsystem to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_tcp_destroy(enyx_tcp * tcp);

/**
 * Get the name of a TCP subsystem.
 *
 * @param tcp The subsystem.
 * @param buffer Output buffer.
 * @param size size of buffer.
 *
 * @return The size of name on success, excluding the terminating null byte
 *         ('\0'). -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_tcp_get_name(enyx_tcp const * tcp, char * buffer, size_t size);

/**
 * Tests if a TCP subsystem has instant ack.
 *
 * @param tcp The subsystem to use.
 * @return True if @p tcp has instant ack. False otherwise.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_has_instant_ack(enyx_tcp const * tcp);

/**
 * Get the maximum segment size supported by a TCP subsystem.
 *
 * @param tcp The subsystem to use.
 * @param mss The mss of @p tcp.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_mss(enyx_tcp const * tcp, uint16_t * mss);

/**
 * Is safe mode enabled on this tcp subsystem.
 *
 * @note If the underlying hardware core is a tcp_multi_stack this parameter is
 * not persistent in the hardware and must be set once for each run of your
 * program (default is disabled).
 *
 * @param tcp The subsystem to use.
 * @return true if safe mode is enabled, false otherwise.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_is_safe_mode_enabled(enyx_tcp const * tcp);

/**
 * Enable safe mode on this subsystem.
 *
 * @note If the underlying hardware core is a tcp_multi_stack this parameter is
 * not persistent in the hardware and must be set once for each run of your
 * program (default is disabled).
 *
 * @param tcp The subsystem to use.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_enable_safe_mode(enyx_tcp * tcp);

/**
 * Disable safe mode on this subsystem.
 *
 * @note If the underlying hardware core is a tcp_multi_stack this parameter is
 * not persistent in the hardware and must be set once for each run of your
 * program (default is disabled).
 *
 * @param tcp The subsystem to use.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_disable_safe_mode(enyx_tcp * tcp);

/**
 * Check if TCP/IP checksum verification is enabled.
 *
 * @param tcp The subsystem to use
 * @return true if checksum verification is enabled, false otherwise
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_is_checksum_verification_enabled(enyx_tcp const * tcp);

/**
 * Enable TCP/IP checksum verification.
 *
 * @param tcp The subsystem to use
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_enable_checksum_verification(enyx_tcp * tcp);

/**
 * Disable TCP/IP checksum verification.
 *
 * @param tcp The subsystem to use
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_disable_checksum_verification(enyx_tcp * tcp);

/**
 * Get SYN retry timeout (in seconds)
 *
 * @param tcp The subsystem to use
 * @param retry_timeout_s The timeout to get
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_syn_retry_timeout(enyx_tcp const * tcp,
                               uint8_t * retry_timeout_s);

/**
 * Set SYN retry timeout (in seconds).
 *
 * @param tcp The subsystem to use
 * @param retry_timeout_s The timeout to set
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 *
 * @warning retry_timeout_s maximum value is 0xF (4 bits).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_set_syn_retry_timeout(enyx_tcp * tcp,
                               uint8_t retry_timeout_s);

/**
 * Check if passive close is enabled.
 *
 * @param tcp The subsystem to use
 * @return true if passive close is enabled, false otherwise
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_is_passive_close_enabled(enyx_tcp const * tcp);

/**
 * Enable passive close.
 *
 * @param tcp The subsystem to use
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_enable_passive_close(enyx_tcp * tcp);

/**
 * Disable passive close.
 *
 * @param tcp The subsystem to use
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_disable_passive_close(enyx_tcp * tcp);

/**
 * Get the Maximum Segment Lifetime (in milliseconds).
 *
 * @param tcp The subsystem to use
 * @param lifetime_ms The lifetime to get
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 *
 * @note TIME_WAIT interval is 2*MSL.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_maximum_segment_lifetime(enyx_tcp const * tcp,
                                      uint32_t * lifetime_ms);

/**
 * Set the Maximum Segment Lifetime (in milliseconds).
 *
 * @param tcp The subsystem to use
 * @param lifetime_ms The lifetime to set
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 *
 * @note TIME_WAIT interval is 2*MSL.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_set_maximum_segment_lifetime(enyx_tcp * tcp,
                                      uint32_t lifetime_ms);

/**
 * Get the size of the RX fifo
 *
 * @param tcp The tcp subsystem to use
 * @param rx_fifo_size The fifo size
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_rx_fifo_size(enyx_tcp const * tcp, uint8_t * rx_fifo_size);

/**
 * Check if the stack will drop outgoing packets on sessions that are not
 * established
 *
 * @param tcp The tcp subsystem to use
 * @return true if enabled
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_drop_tx_packets_if_not_established_enabled(enyx_tcp const * tcp);

/**
 * Check if the OOS_SEQNUM feature is enabled.
 *
 * @param tcp The tcp subsystem to use
 * @return true if enabled
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_oos_seqnum_enabled(enyx_tcp const * tcp);

/**
 * Check if status events are sent to the software.
 *
 * @param tcp The tcp subsystem to use
 * @return true if enabled
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_status_events_enabled(enyx_tcp const * tcp);

/**
 * Check if TX Credit events are sent to the software.
 *
 * @param tcp The tcp subsystem to use
 * @return true if enabled
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_credit_events_enabled(enyx_tcp const * tcp);

/**
 * Check if retransmission External Memory is enabled.
 *
 * @param tcp The tcp subsystem to use
 * @return true if enabled
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_retransmission_external_memory_enabled(enyx_tcp const * tcp);

/**
 * Check if retransmission external full duplex memory is enabled.
 *
 * @param tcp The tcp subsystem to use
 * @return true if enabled
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_retransmission_full_duplex_external_memory_enabled(enyx_tcp const * tcp);

/**
 * Get retransmission external memory read latency.
 *
 * @param tcp The tcp subsystem to use
 * @param value The latency
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_retransmission_external_memory_latency(enyx_tcp const * tcp,
                                                    uint8_t * value);
/**
 * Get retransmission external/internal memory address width.
 *
 * @param tcp The tcp subsystem to use
 * @param value the address width
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_retransmission_memory_address_width(enyx_tcp const * tcp,
                                                 uint8_t * value);

/**
 * Get the retransmission external/internal memory data width.
 *
 * @param tcp The tcp subsystem to use
 * @param value The retransmission data width
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_retransmission_memory_data_width(enyx_tcp const * tcp,
                                              uint16_t * value);

/**
 * Check if reordering is enabled.
 *
 * @param tcp The tcp subsystem to use
 * @return true if enabled
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_tcp_reordering_enabled(enyx_tcp const * tcp);

/**
 * Get the reordering memory address width.
 *
 * @param tcp The tcp subsystem to use
 * @param value The reordering address width
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_reordering_memory_address_width(enyx_tcp const * tcp,
                                             uint8_t * value);

/**
 * Get the reordering memory data width.
 *
 * @param tcp The tcp subsystem to use
 * @param value The reordering memory data width
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_reordering_memory_data_width(enyx_tcp const * tcp,
                                          uint16_t * value);

/**
 * Get the number of sessions supported by a TCP subsystem.
 *
 * @param tcp The subsystem to use.
 * @return The number of sessions.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY uint16_t
enyx_tcp_get_session_count(enyx_tcp const * tcp);

/**
 * Get a session object from a TCP subsystem.
 *
 * @param tcp The tcp subsystem to use.
 * @param session_id The id of the requested session.
 * @param session The session object
 *
 * @note The @p session needs the @p tcp object to work, if you delete the @p
 * tcp object, using the @p session is undefined behavior.
 *
 * @return 0 on success, -1 on failure (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_session(enyx_tcp const * tcp, uint16_t session_id,
                     enyx_tcp_session * session);

/**
 * Get a session that is currently closed.
 *
 * @note
 * The get_available_session() is a specific function designed simply to
 * get the first TCP session that currently has the "CLOSED" status.
 * If you call `get_available_session()` several times without opening the
 * session it will provide the same session every time.  If a session fails to
 * open and enters "CLOSED" status again, it will go back to the available pool
 * and may be obtained again.
 * If you need to get several sessions, we suggest using a combination of
 * `get_session()` and either the `session_monitor` object or the data stream
 * `context` object to monitor the sessions.
 *
 * @param tcp The subsystem to use.
 * @param session The session to be selected.
 * @return 0 if a session was found. -1 on error (with @b errno
 *         set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_available_session(enyx_tcp const * tcp,
                               enyx_tcp_session * session);

/**
 * A structure representing the stack runtime statistics.
 */
typedef struct {
    /// Number of packets with TCP checksum errors, for all sessions
    uint32_t rx_tcp_checksum_errors;
    /// The number of packets dropped because of a reception FIFO overflow
    uint32_t rx_fifo_backressure;
    /// The maximum capacity of the RX FIFO in packets.
    uint32_t rx_fifo_maximum_packet_count;
    /// The amount of packets currently in the RX FIFO
    uint32_t rx_fifo_used_packet_count;
    /// The maximum capacity of the RX FIFO in words
    uint32_t rx_fifo_max_available_words;
    /// The amount of words currently in the RX FIFO
    uint32_t rx_fifo_used_words;
    /// Number of times the re-ordering mechanism has been triggered
    uint32_t rx_reordering_count;
    /// Number of recovered packets that had been lost
    uint32_t rx_reordering_recovered_packet_count;
    /**
     * Total number of packets stored inside the Rx re-ordering buffer for
     * later recovery
     */
    uint32_t rx_reordering_stored_packet_count;
    /**
     * Number of packets that could have been stored in the Rx re-ordering
     * buffer space but are dropped because the sequence number exceeds the
     * buffer space allocated.
     */
    uint32_t rx_reordering_out_of_buffer_packet_count;
    /// Number of times a second gap is detected in the re-ordering mechanism
    uint32_t rx_reordering_multigap_packet_count;
} enyx_tcp_statistics;

/// @cond
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_get_statistics_(enyx_tcp const * tcp,
                         enyx_tcp_statistics * statistics,
                         size_t statistics_size);
/// @endcond

/**
 * Get the TCP subsystem Hardware Statistics
 *
 * @param tcp The tcp subsystem
 * @param statistics The statistics object
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
#define enyx_tcp_get_statistics(tcp, statistics) \
    enyx_tcp_get_statistics_(tcp, statistics, sizeof(*statistics))
