#pragma once

#include <stdbool.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/symbol_visibility.h>

typedef struct enyx_raw_ethernet enyx_raw_ethernet;

/**
 * Construct the RAW ETHERNET module
 *
 * This module configures the following hw core:
 *
 * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
 * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
 *
 * @param core The hardware core containing the stack.
 * @note This requires you to find a compatible enyx_hw_core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_raw_ethernet *
enyx_raw_ethernet_create(enyx_hw_core * core);

/**
 * Destroy the RAW ETHERNET module.
 *
 * @param raw_ethernet The raw_ethernet subsystem to destroy
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_raw_ethernet_destroy(enyx_raw_ethernet * raw_ethernet);

/**
 * Get the MAC data width.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @param width The mac data with
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_raw_ethernet_get_mac_data_width(const enyx_raw_ethernet * raw_ethernet,
                                     uint8_t * width);

/**
 * Get the raw_ethernet data width.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @param width The data with
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_raw_ethernet_get_data_width(const enyx_raw_ethernet * raw_ethernet,
                                 uint8_t * width);

/**
 * Check is the raw_ethernet capture mode is enabled.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @return The raw_ethernet capture mode activation status
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_raw_ethernet_capture_enabled(const enyx_raw_ethernet * raw_ethernet);

/**
 * Enable raw_ethernet capture mode.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_raw_ethernet_enable_capture(enyx_raw_ethernet * raw_ethernet);

/**
 * Disable raw_ethernet capture mode.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_raw_ethernet_disable_capture(enyx_raw_ethernet * raw_ethernet);

/**
 * Check if the raw_ethernet RX net_interface is available.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @return The raw_ethernet RX net_interface availability
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_raw_ethernet_rx_interface_available(const enyx_raw_ethernet * raw_ethernet);

/**
 * Get the raw_ethernet RX total packet count.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @param count The packet count
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_raw_ethernet_get_rx_packet_count(const enyx_raw_ethernet *raw_ethernet,
                                      uint32_t * count);

/**
 * Get the raw_ethernet RX packet count in error state.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @param count The packet count
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_raw_ethernet_get_rx_error_count(const enyx_raw_ethernet *raw_ethernet,
                                     uint32_t * count);

/**
 * Check if the raw_ethernet TX net_interface is available.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @return The raw_ethernet TX net_interface availability
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_raw_ethernet_tx_interface_available(const enyx_raw_ethernet * raw_ethernet);

/**
 * Get the raw_ethernet TX total packet count.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @param count The packet count
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_raw_ethernet_get_tx_packet_count(const enyx_raw_ethernet *raw_ethernet,
                                      uint32_t * count);

/**
 * Get the raw_ethernet TX packet count in error state.
 *
 * @param raw_ethernet The raw_ethernet subsystem
 * @param count The packet count
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_raw_ethernet_get_tx_error_count(const enyx_raw_ethernet *raw_ethernet,
                                     uint32_t * count);
