/**
 * @file
 *
 * Contains the types and functions related to the probe.
 */
#pragma once

#include <stdint.h>
#include <stdbool.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/probes/event.h>

/**
 * Represents an probes probe
 *
 * This class can be used to query the probe and configure it.
 *
 * @since 6.7.0
 */
typedef struct enyx_probes_probe enyx_probes_probe;

/// The probe id
typedef uint8_t enyx_probes_probe_id;

/**
 * Create a probe.
 *
 * @param core The probe core to create from.
 * @param new_id The probe ID.
 * @param ts_format The probe event format.
 * @return The probe in case of success, NULL otherwise (with errno set
 * accordingly).
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_probe *
enyx_probes_probe_create(enyx_hw_core * core, uint8_t new_id,
                         enyx_probes_ts_format ts_format);

/**
 * Destroy a probe.
 *
 * @param probe The probe to destroy.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_probes_probe_destroy(enyx_probes_probe * probe);

/**
 * Get the probe event format.
 *
 * @param probe probe to query.
 * @return 0 on success, -1 with errno set on failure.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_probe_event_format const*
enyx_probes_probe_get_event_format(enyx_probes_probe const * probe);

/**
 * Get the probe name.
 *
 * @param probe probe to query.
 * @return a pointer to the probe name string.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY char const*
enyx_probes_probe_get_name(enyx_probes_probe const* probe);

/**
 * Get the associated collector name.
 *
 * @param probe probe to query.
 * @return a pointer to the probe name string.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY char const*
enyx_probes_probe_get_collector_name(enyx_probes_probe const* probe);

/**
 * Get the probe id.
 *
 * @param probe probe to query.
 * @param id the probe identifier previously set.
 * @return 0 on success or -1 on failure (with @b errno set accordingly).
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_probes_probe_get_id(enyx_probes_probe const * probe,
                         enyx_probes_probe_id * id);

/**
 * Get the probe status.
 *
 * @param probe probe to query.
 * @return a boolean indicating the probe status.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_probes_probe_is_enabled(enyx_probes_probe const * probe);

/**
 * Enable the probe.
 *
 * @param probe probe to set.
 * @return 0 on success, -1 with errno set on failure.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_probes_probe_enable(enyx_probes_probe const* probe);

/**
 * Disable the probe.
 *
 * @param probe probe to set.
 * @return 0 on success, -1 with errno set on failure.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_probes_probe_disable(enyx_probes_probe const* probe);

/**
 * Get the probe dropped event(s) count.
 *
 * @param probe probe to query.
 * @return the count on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_probes_probe_get_drop_count(enyx_probes_probe const * probe);

/**
 * Get the probe event packet(s) count.
 *
 * @param probe probe to query.
 * @return the count on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_probes_probe_get_packet_count(enyx_probes_probe const * probe);
