#pragma once

#include <stdbool.h>

#include <enyx/cores_c/symbol_visibility.h>

#include <enyx/hw_c/core.h>

typedef struct enyx_ethernet_filter enyx_ethernet_filter;

/**
 * Construct the ETHERNET FILTER module
 *
 * This module configures the following hw core:
 *
 * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
 * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
 *
 * @param core The hardware core containing the stack.
 * @note This requires you to find a compatible hw::core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_ethernet_filter *
enyx_ethernet_filter_create(enyx_hw_core * core);

/**
 * Destroy the ethernet_filter subsystem.
 *
 * @param ethernet_filter The ethernet_filter subsystem to destroy
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_ethernet_filter_destroy(enyx_ethernet_filter * ethernet_filter);
/**
 * Check is the ethernet_filter is enabled.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @return The ethernet_filter activation status
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_ethernet_filter_is_enabled(enyx_ethernet_filter const * ethernet_filter);

/**
 * Enable ethernet_filter.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_enable(enyx_ethernet_filter * ethernet_filter);

/**
 * Disable ethernet_filter.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_disable(enyx_ethernet_filter * ethernet_filter);

/**
 * Check if unicast promiscuous mode is enabled.
 * When enabled, do not check Unicast MAC, forward all packets.
 *
 * Default disabled.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @return The unicast promiscuous activation status
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_ethernet_filter_unicast_promiscuous_enabled(enyx_ethernet_filter const * ethernet_filter);

/**
 * Enable unicast promiscuous mode.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_enable_unicast_promiscuous(enyx_ethernet_filter * ethernet_filter);

/**
 * Disable unicast promiscuous mode.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_disable_unicast_promiscuous(enyx_ethernet_filter * ethernet_filter);

/**
 * Check if UDP session copy to filter is enabled.
 *
 * When enabled, all packets from this UDP session will be copied to filter.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @param udp_session_id The UDP session to check
 * @param enabled True if the UDP session copy is enabled, false otherwise
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_udp_session_copy_enabled(
        enyx_ethernet_filter const * ethernet_filter,
        uint16_t udp_session_id,
        bool * enabled);

/**
 * Enable UDP session copy to filter.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @param udp_session_id The UDP session to enable
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_enable_udp_session_copy(
        enyx_ethernet_filter * ethernet_filter, uint16_t udp_session_id);

/**
 * Disable UDP session copy to filter.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @param udp_session_id The UDP session to disable
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_disable_udp_session_copy(
        enyx_ethernet_filter * ethernet_filter, uint16_t udp_session_id);

/**
 * Get the number of packet before applying the ethernet_filter.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @param count The packet count
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_get_packet_fifo_out_count(
        enyx_ethernet_filter const *ethernet_filter, uint32_t *count);

/**
 * Get the number of packet dropped by the ethernet_filter.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @param count The packet count
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_get_packet_filter_drop_count(
        enyx_ethernet_filter const *ethernet_filter, uint32_t *count);

/**
 * Get the number of packet after applying the ethernet_filter.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @param count The packet count
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_get_packet_filter_out_count(
        enyx_ethernet_filter const *ethernet_filter, uint32_t *count);

/**
 * Get the filtered data width.
 *
 * @param ethernet_filter The ethernet_filter subsystem
 * @param width The mac data with
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_filter_get_filtered_data_width(
        enyx_ethernet_filter const *ethernet_filter, uint8_t * width);
