ENYX_CORES_NAMESPACE_BEGIN

namespace uart {

inline
uart::uart(enyx::hw::core const & subtree)
    : subtree_root_(subtree)
    , uart_(::enyx_uart_create(subtree_root_.handle()))
{
    if (uart_ == nullptr)
        throw std::system_error{errno, std::generic_category(),
            "Failed to create UART subsystem"};
}

inline result<std::uint32_t>
uart::get_clock_frequency() const noexcept
{
    std::uint32_t frequency;
    if (enyx_uart_get_clock_frequency(handle(), &frequency) < 0)
        return std::error_code{errno, std::generic_category()};

    return frequency;
}

inline result<std::uint32_t>
uart::get_baudrate() const noexcept
{
    std::uint32_t baudrate;
    if (enyx_uart_get_baudrate(handle(), &baudrate) < 0)
        return std::error_code{errno, std::generic_category()};

    return baudrate;
}

inline result<bool>
uart::is_rx_empty() const noexcept
{
    bool empty;
    if (enyx_uart_is_rx_empty(handle(), &empty) < 0)
        return std::error_code{errno, std::generic_category()};

    return empty;
}

inline result<bool>
uart::is_tx_full() const noexcept
{
    bool full;
    if (enyx_uart_is_tx_full(handle(), &full) < 0)
        return std::error_code{errno, std::generic_category()};

    return full;
}

inline result<bool>
uart::is_rx_busy() const noexcept
{
    bool busy;
    if (enyx_uart_is_rx_busy(handle(), &busy) < 0)
        return std::error_code{errno, std::generic_category()};

    return busy;
}

inline result<bool>
uart::is_tx_busy() const noexcept
{
    bool busy;
    if (enyx_uart_is_tx_busy(handle(), &busy) < 0)
        return std::error_code{errno, std::generic_category()};

    return busy;
}

inline result<void>
uart::set_baudrate_divisor(std::uint16_t divisor) noexcept
{
    if (enyx_uart_set_baudrate_divisor(handle(), divisor))
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<void>
uart::clear_rx_fifo() noexcept
{
    if (enyx_uart_clear_rx_fifo(handle()))
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<void>
uart::clear_tx_fifo() noexcept
{
    if (enyx_uart_clear_tx_fifo(handle()))
        return std::error_code{errno, std::generic_category()};

    return {};
}

inline result<std::size_t>
uart::read(std::uint8_t * data, std::size_t size) noexcept
{
    auto res = enyx_uart_read(handle(), data, size);
    if (res < 0)
        return std::error_code{errno, std::generic_category()};

    return res;
}

inline result<std::size_t>
uart::read_all(std::uint8_t * data, std::size_t size,
               std::uint64_t timeout_ms) noexcept
{
    auto res = enyx_uart_read_all(handle(), data, size, timeout_ms);
    if (res < 0)
        return std::error_code{errno, std::generic_category()};

    return res;
}

inline result<size_t>
uart::write(const std::uint8_t * data, std::size_t size) noexcept
{
    auto res = enyx_uart_write(handle(), data, size);
    if (res < 0)
        return std::error_code{errno, std::generic_category()};

    return res;
}

inline result<std::size_t>
uart::write_all(const std::uint8_t * data, std::size_t size,
                std::uint64_t timeout_ms) noexcept
{
    auto res = enyx_uart_write_all(handle(), data, size, timeout_ms);
    if (res < 0)
        return std::error_code{errno, std::generic_category()};

    return res;
}

inline ::enyx_uart const *
uart::handle() const noexcept
{
    return uart_.get();
}

inline ::enyx_uart *
uart::handle() noexcept
{
    return uart_.get();
}

} /* namespace uart */

ENYX_CORES_NAMESPACE_END
