ENYX_CORES_NAMESPACE_BEGIN
namespace tcp {

inline
tcp::tcp(enyx::hw::core const& tcp_core)
    : tcp_core_(tcp_core),
      tcp_(::enyx_tcp_create(tcp_core_.handle()),
           ::enyx_tcp_destroy)

{
    if (! tcp_)
        throw std::system_error{errno, std::generic_category(),
                                "Failed to create tcp"};
}

inline enyx::hw::core
tcp::get_core() const noexcept
{
    return tcp_core_;
}

inline result<std::string>
tcp::get_name() const noexcept
{
    char buffer[128];
    ssize_t ret = ::enyx_tcp_get_name(handle(),
                                          buffer,
                                          sizeof(buffer));
    if (ret < 0)
        return std::error_code{errno, std::generic_category()};

    return std::string{buffer};
}

inline bool
tcp::has_instant_ack() const noexcept
{
    return ::enyx_tcp_has_instant_ack(handle());
}

inline result<std::uint16_t>
tcp::get_mss() const noexcept
{
    std::uint16_t mss;
    if (::enyx_tcp_get_mss(handle(), &mss) != 0)
        return std::error_code{errno, std::generic_category()};

    return mss;
}

inline bool
tcp::is_safe_mode_enabled() const noexcept
{
    return ::enyx_tcp_is_safe_mode_enabled(handle());
}

inline result<void>
tcp::enable_safe_mode() noexcept
{
    if (::enyx_tcp_enable_safe_mode(handle()) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline result<void>
tcp::disable_safe_mode() noexcept
{
    if (::enyx_tcp_disable_safe_mode(handle()) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline std::uint16_t
tcp::get_session_count() const noexcept
{
    return ::enyx_tcp_get_session_count(handle());
}

inline result<session>
tcp::get_session(std::uint16_t session_id) const noexcept
{
    ::enyx_tcp_session c_session;
    if (::enyx_tcp_get_session(handle(), session_id, &c_session) != 0)
        return std::error_code{errno, std::generic_category()};
    return session{tcp_, c_session};
}

inline result<session>
tcp::get_available_session() const noexcept
{
    ::enyx_tcp_session c_session;
    if (::enyx_tcp_get_available_session(handle(), &c_session) != 0)
        return std::error_code{errno, std::generic_category()};
    return session{tcp_, c_session};
}

inline bool
tcp::is_checksum_verification_enabled() const noexcept
{
    return ::enyx_tcp_is_checksum_verification_enabled(handle());
}

inline result<void>
tcp::enable_checksum_verification()
{
    if (::enyx_tcp_enable_checksum_verification(handle()) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline result<void>
tcp::disable_checksum_verification()
{
    if (::enyx_tcp_disable_checksum_verification(handle()) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline result<std::uint8_t>
tcp::get_syn_retry_timeout() const noexcept
{
    std::uint8_t value;
    if (::enyx_tcp_get_syn_retry_timeout(handle(), &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline result<void>
tcp::set_syn_retry_timeout(std::uint8_t retry_timeout_s)
{
    if (::enyx_tcp_set_syn_retry_timeout(handle(), retry_timeout_s) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline bool
tcp::is_passive_close_enabled() const noexcept
{
    return ::enyx_tcp_is_passive_close_enabled(handle());
}

inline result<void>
tcp::enable_passive_close()
{
    if (::enyx_tcp_enable_passive_close(handle()) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline result<void>
tcp::disable_passive_close()
{
    if (::enyx_tcp_disable_passive_close(handle()) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline result<std::uint32_t>
tcp::get_maximum_segment_lifetime() const noexcept
{
    std::uint32_t value;
    if (::enyx_tcp_get_maximum_segment_lifetime(handle(), &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline result<void>
tcp::set_maximum_segment_lifetime(std::uint32_t lifetime_ms)
{
    if (::enyx_tcp_set_maximum_segment_lifetime(handle(),
                                                    lifetime_ms) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline result<std::uint8_t>
tcp::get_rx_fifo_size() const noexcept
{
    std::uint8_t value;
    if (::enyx_tcp_get_rx_fifo_size(handle(), &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline bool
tcp::drop_tx_packets_if_not_established_enabled() const noexcept
{
    return ::enyx_tcp_drop_tx_packets_if_not_established_enabled(handle());
}

inline bool
tcp::oos_seqnum_enabled() const noexcept
{
    return ::enyx_tcp_oos_seqnum_enabled(handle());
}

inline bool
tcp::status_events_enabled() const noexcept
{
    return ::enyx_tcp_status_events_enabled(handle());
}

inline bool
tcp::credit_events_enabled() const noexcept
{
    return ::enyx_tcp_credit_events_enabled(handle());
}

inline bool
tcp::retransmission_external_memory_enabled() const noexcept
{
    return ::enyx_tcp_retransmission_external_memory_enabled(handle());
}

inline bool
tcp::retransmission_full_duplex_external_memory_enabled() const noexcept
{
    return ::enyx_tcp_retransmission_full_duplex_external_memory_enabled(handle());
}

inline result<std::uint8_t>
tcp::get_retransmission_external_memory_latency() const noexcept
{
    std::uint8_t value;
    if (::enyx_tcp_get_retransmission_external_memory_latency(handle(), &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline result<std::uint8_t>
tcp::get_retransmission_memory_address_width() const noexcept
{
    std::uint8_t value;
    if (::enyx_tcp_get_retransmission_memory_address_width(handle(), &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline result<std::uint16_t>
tcp::get_retransmission_memory_data_width() const noexcept
{
    std::uint16_t value;
    if (::enyx_tcp_get_retransmission_memory_data_width(handle(), &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline bool
tcp::reordering_enabled() const noexcept
{
    return ::enyx_tcp_reordering_enabled(handle());
}

inline result<std::uint8_t>
tcp::get_reordering_memory_address_width() const noexcept
{
    std::uint8_t value;
    if (::enyx_tcp_get_reordering_memory_address_width(handle(), &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline result<std::uint16_t>
tcp::get_reordering_memory_data_width() const noexcept
{
    std::uint16_t value;
    if (::enyx_tcp_get_reordering_memory_data_width(handle(), &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline result<statistics>
tcp::get_statistics() const noexcept
{
    statistics ret{};
    if (::enyx_tcp_get_statistics(handle(), &ret) != 0)
        return std::error_code{errno, std::generic_category()};
    return ret;
}

inline ::enyx_tcp *
tcp::handle() const noexcept
{
    return tcp_.get();
}

} /* namespace tcp */
ENYX_CORES_NAMESPACE_END
