/**
 *  @file
 *
 *  Contains the types and functions related to the timestamping.
 */
#pragma once

#include <enyx/cores_c/probes/timestamp.h>

#include <cstdio>

#include <iomanip>
#include <iostream>

#include <enyx/cores/namespace.hpp>

ENYX_CORES_NAMESPACE_BEGIN

namespace probes {

/// @copydoc enyx_probes_ts_format
using ts_format = ::enyx_probes_ts_format;

/**
 * Print the timestamp format @p ts into the output stream @p out
 *
 * @param out The output stream
 * @param format The timestamp format to print
 * @return A reference to @p out
 *
 * @since 6.7.0
 */
std::ostream &
operator<<(std::ostream & out, ts_format const& format);

/// @copydoc enyx_probes_duration
using duration = ::enyx_probes_duration;

/**
 * Print the duration @p d into the output stream @p out
 *
 * @param out The output stream
 * @param d The duration to print
 * @return A reference to @p out
 *
 * @since 6.7.0
 */
std::ostream &
operator<<(std::ostream & out, duration const& d);

/// @copydoc enyx_probes_ts_64_v1
using ts_64_v1 = ::enyx_probes_ts_64_v1;

/// @copydoc enyx_probes_ts_parse_64_v1
ts_64_v1
ts_parse_64_v1(void const* data);

/// @copydoc enyx_probes_ts_96_v1_to_duration
duration
operator-(ts_64_v1 const& a, ts_64_v1 const& b);

/**
 * Print the timestamp into the output stream @p out
 *
 * The format is 's.ns'.
 *
 * @param out The output stream
 * @param ts The timestamp to print
 * @return A reference to @p out
 *
 * @since 6.7.0
 */
std::ostream &
operator<<(std::ostream & out, ts_64_v1 const& ts);

/// @copydoc enyx_probes_ts_96_v1
using ts_96_v1 = ::enyx_probes_ts_96_v1;

/// @copydoc enyx_probes_ts_parse_96_v1
ts_96_v1
ts_parse_96_v1(void const* data);

/// @copydoc enyx_probes_ts_96_v1_to_duration
duration
operator-(ts_96_v1 const& a, ts_96_v1 const& b);

/**
 * Print the timestamp into the output stream @p out
 *
 * The format is 's.ns'.
 *
 * @param out The output stream
 * @param ts The timestamp to print
 * @return A reference to @p out
 *
 * @since 6.7.0
 */
std::ostream &
operator<<(std::ostream & out, ts_64_v1 const& ts);

} /* namespace probes */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/probes/timestamp.ipp>

// Koening lookup is of no-use here as the types
// are defined in the :: scope, and only brought
// here by 'using cxx_name = ::c_name;'
// So if the types can't be moved into enyx::probes,
// move the operator<< into :: scope.
using enyx::probes::operator<<;
using enyx::probes::operator-;
