/**
 *  @file
 *
 *  Contains the types and functions related to the system.
 */
#pragma once

#include <enyx/cores_c/probes/system.h>

#include <cerrno>

#include <iostream>
#include <memory>
#include <system_error>
#include <vector>

#include <enyx/hw/result.hpp>
#include <enyx/hw/core.hpp>

#include <enyx/cores/namespace.hpp>

#include <enyx/cores/probes/collector.hpp>
#include <enyx/cores/probes/mtg.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_probes_system>
{
    void
    operator()(::enyx_probes_system * ptr) const
    {
        ::enyx_probes_system_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace probes {

/**
 * @copydoc enyx_probes_system
 *
 * @since 6.7.0
 */
class system final
{
public:
    /// Represents a sequence of event collectors
    using collectors = std::vector<collector>;
    //
    /// The associated probes sequence
    using probes = std::vector<probe>;

/// @cond
public:
    friend std::ostream &
    operator<<(std::ostream & out, system const& s);
/// @endcond

public:
    /**
     * Construct a system from a root core.
     *
     * @param root The root core
     *
     * @since 6.7.0
     */
    system(hw::core const & root);

    /**
     * Direct access to the underlying C enyx_probes_system object.
     *
     * @return The C enyx_probes_system object.
     *
     * @since 6.7.0
     */
    ::enyx_probes_system *
    handle() noexcept;

    /**
     * Direct access to the underlying C enyx_probes_system object.
     *
     * @return The C enyx_probes_system object.
     *
     * @since 6.7.0
     */
    ::enyx_probes_system const *
    handle() const noexcept;

    /**
     * @copybrief enyx_probes_system_enumerate_collectors
     *
     * @return A container filled with the enumerated event collectors
     *
     * @since 6.7.0
     */
    result<collectors>
    enumerate_collectors() const noexcept;

    /**
     *  Retrieve the associated probes.
     *
     *  @return The associated probes.
     *
     * @since 6.7.0
     */
    result<probes>
    enumerate_probes(collector const & collector);

    /**
     * @copybrief enyx_probes_system_get_mtg
     *
     * @return The system's mtg instance
     *
     * @since 6.7.0
     */
    result<mtg>
    get_mtg() const noexcept;

private:
    hw::core root_;
    std::unique_ptr<::enyx_probes_system> system_;
};

/**
 * Print the system @p m into the output stream @p out
 *
 * @param out The output stream
 * @param p The system to print
 * @return A reference to @p out
 *
 * @since 6.7.0
 */
std::ostream &
operator<<(std::ostream & out, system const& p);

} /* namespace probes */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/probes/system.ipp>
