#pragma once

#include <memory>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/hw/core.hpp>

#include <enyx/cores_c/i2c/i2c.h>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_i2c_bus>
{
    void
    operator()(::enyx_i2c_bus * ptr) const
    {
        ::enyx_i2c_bus_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace i2c {

class bus;

/**
 * @copydoc enyx_i2c
 */
class i2c {
public:
    /**
     * Construct an I2C object.
     *
     * @param core The root core.
     *
     * @throw std::system_error on failure.
     */
    i2c(enyx::hw::core & core);

    /**
     * Get the count of I2C bus
     *
     * @return A result containing the bus count or an error.
     */
    result<size_t>
    get_bus_count();

    /**
     * Get an I2C bus from its ID.
     *
     * @return A result containing the bus ID or an error.
     */
    result<bus>
    get_bus(uint8_t bus);

    /**
     * Read I2C status pins
     *
     * @note The number of pins and the meaning of each pin is board specific.
     *
     * @return A result containing a vector of status pin values on success or
     * an error on failure.
     */
    result<std::vector<bool>>
    read_status_pins();

    /**
     * Read I2C control pins
     *
     * @note The number of pins and the meaning of each pin is board specific.
     *
     * @return A result containing a vector of control pin values on success or
     * an error on failure.
     */
    result<std::vector<bool>>
    read_control_pins();

    /**
     * Write I2C control pins
     *
     * @note The number of pins and the meaning of each pin is board specific.
     *
     * @param values A vector of control pin values to set
     * @return An error on failure
     */
    result<void>
    write_control_pins(std::vector<bool> values);

    /**
     * Get a const pointer to the underlying @b C handle @ref enyx_i2c.
     *
     * @return A const pointer to a @ref enyx_i2c.
     */
    enyx_i2c const *
    handle() const noexcept;

    /**
     * Get a pointer to the underlying @b C handle @ref enyx_i2c.
     *
     * @return A pointer to a @ref enyx_i2c.
     */
    enyx_i2c *
    handle() noexcept;

private:
    std::shared_ptr<enyx_i2c> i2c_c_;
};

/**
 * @copydoc enyx_i2c_bus
 */
class bus {
public:
    /**
     * Construct an I2C bus object.
     *
     * @note This should not be used directly. i2c::get_bus should be used
     * instead.
     *
     * @param i2c_c A shared pointer to the underlying ::enyx_i2c to use
     * @param bus_c The ::enyx_i2c_bus object to construct from
     */
    bus(std::shared_ptr<::enyx_i2c> i2c_c, ::enyx_i2c_bus * bus_c);

    /**
    * Read from an I2C device.
    *
    * @param address The I2C device address
    * @param data A buffer to fill with the read data.
    * @param size The size to read from the device.
    */
    result<size_t>
    read(const uint8_t address, uint8_t * data, size_t size);

    /**
    * Write to an I2C device.
    *
    * @param address The I2C device address
    * @param data A buffer containing the data to write.
    * @param size The data buffer size.
    */
    result<void>
    write(const uint8_t address, uint8_t const * data, size_t size);

    /**
    * Read a register from an I2C device.
    *
    * @param address The I2C device address
    * @param reg The device register address.
    * @param data A buffer to fill with the register data.
    * @param size The size of the read register.
    */
    result<size_t>
    read_register(const uint8_t address, const uint8_t reg,
                uint8_t * data, size_t size);

    /**
    * Write to an I2C device register.
    *
    * @param address The I2C device address
    * @param reg The device register address.
    * @param data A buffer containing the data to write to the register.
    * @param size The data buffer size.
    */
    result<void>
    write_register(const uint8_t address, const uint8_t reg,
                uint8_t const * data, size_t size);

    /** Set the I2C bus clock prescale.
    *
    * @param clock_prescale The clock prescale.
    * @return A result containing an error in case of error.
    */
    result<void>
    set_clock_prescale(enyx_i2c_clock_prescale clock_prescale);

    /**
    * Set the I2C bus read timeout.
    *
    * @param timeout_ms The timeout is miliseconds.
    */
    result<void>
    set_read_timeout(uint64_t timeout_ms);

    /**
    * Get a const pointer to the underlying @b C handle @ref enyx_i2c_bus.
    *
    * @return A const pointer to a @ref enyx_i2c_bus.
    */
    enyx_i2c_bus const *
    handle() const noexcept;

    /**
    * Get a pointer to the underlying @b C handle @ref enyx_i2c_bus.
    *
    * @return A pointer to a @ref enyx_i2c_bus.
    */
    enyx_i2c_bus *
    handle() noexcept;

private:
    std::shared_ptr<::enyx_i2c> i2c_c_;
    std::unique_ptr<::enyx_i2c_bus> bus_c_;
};

} /* namespace i2c */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/i2c/i2c.ipp>
