#pragma once

#include <array>

#include <enyx/cores_c/hw_top/hw_top.h>
#include <enyx/cores/macros.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/types.hpp>
#include <enyx/cores/hw_top/serial_id.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/hw/core.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_hw_top>
{
    void
    operator()(::enyx_hw_top * ptr) const
    {
        ::enyx_hw_top_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace hw_top {

/**
 * Main HW top object to manage the @b hw_top core.
 */
class hw_top {
public:
    /**
     * Construct a HW top object from a core subtree.
     *
     * Search through the given core subtree to find the HW top core and create
     * a HW top object from it.
     *
     * @param subtree The core subtree to use.
     * @throw std::system_error on error.
     */
    hw_top(enyx::hw::core const & subtree);

    /**
     * @copybrief enyx_hw_top_get_build_date
     *
     * @return A result containing the build date or an error.
     */
    result<time_t>
    get_build_date() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_seed
     *
     * @return A result containing the seed or an error.
     */
    result<std::uint32_t>
    get_seed() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_fpga_uptime
     *
     * @return A result containing the FPGA uptime or an error.
     */
    result<std::uint32_t>
    get_fpga_uptime() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_fpga_serial_id
     *
     * @return A result containing the FPGA serial ID or an error.
     */
    result<serial_id>
    get_fpga_serial_id() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_temperature
     *
     * @return A result containing the temperature or an error.
     */
    result<double>
    get_temperature() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_memories_init_status
     *
     * @return A result containing the memories init satus or an error.
     */
    result<std::uint32_t>
    get_memories_init_status() const noexcept;

    /**
     * @deprecated
     * Use @ref get_firmware_identifier instead.
     *
     * @copybrief enyx_hw_top_get_unique_identifier
     *
     * @return A result containing the unique identifier or an error.
     */
    ENYX_CORES_CXX_DEPRECATED("replaced by get_firmware_identifier")
    result<std::uint64_t>
    get_unique_identifier() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_firmware_identifier
     *
     * @return A result containing the firmware identifier or an error.
     */
    result<std::string>
    get_firmware_identifier() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_stream_count
     *
     * @return A result containing the stream count or an error.
     */
    result<std::uint8_t>
    get_stream_count() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_memory_count
     *
     * @return A result containing the memory count or an error.
     */
    result<std::uint8_t>
    get_memory_count() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_clock_count
     *
     * @return A result containing the clock count or an error.
     */
    result<std::uint8_t>
    get_clock_count() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_clock_frequency
     *
     * @return A result containing the clock frequency or an error.
     */
    result<std::uint32_t>
    get_clock_frequency() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_mac_addresses
     *
     * @return A result containing a vector of MAC addresses or an error.
     */
    std::vector<mac_address>
    get_mac_addresses() const noexcept;

    /**
     * @deprecated
     * Board serial ID retrieval through HW top is not supported anymore.
     *
     * @copybrief enyx_hw_top_get_board_serial_id
     *
     * @return A result containing the board serial ID or an error.
     */
    ENYX_CORES_CXX_DEPRECATED("Board serial ID retrieval through HW top is not supported anymore")
    result<std::uint32_t>
    get_board_serial_id() const noexcept;

    /**
     * @copybrief enyx_hw_top_reset_global
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    reset_global() noexcept;

    /**
     * @copybrief enyx_hw_top_reset_module
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    reset_module() noexcept;

    /**
     * @copybrief enyx_hw_top_reset_phy
     *
     * @param mask The interfaces to reset mask.
     * @return A result object containing an error on failure.
     */
    result<void>
    reset_phy(uint64_t mask) noexcept;

    /**
     * @copybrief enyx_hw_top_update_params
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    update_params() noexcept;

    /**
     * @copybrief enyx_hw_top_get_top_name
     *
     * @return A result containing the top name string or an error.
     */
    result<std::string>
    get_top_name() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_product_id
     *
     * @return A result containing the product ID or an error.
     */
    result<std::string>
    get_product_id() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_build_host
     *
     * @return A result containing the build hostname string or an error.
     */
    result<std::string>
    get_build_host() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_build_tool
     *
     * @return A string containing the build tool description  or an error.
     */
    result<std::string>
    get_build_tool() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_device_id
     *
     * @return A string containing the device ID or an error.
     */
    result<std::string>
    get_device_id() const noexcept;

    /**
     * @copybrief enyx_hw_top_get_board_name
     *
     * @return A string containing the board name or an error.
     */
    result<std::string>
    get_board_name() const noexcept;

    /**
     * Get a pointer to the underlying @b C handle @ref enyx_hw_top.
     *
     * @return A pointer to a @ref enyx_hw_top.
     */
    enyx_hw_top *
    handle() noexcept;

    /**
     * Get a pointer to the underlying @b C handle @ref enyx_hw_top.
     *
     * @return A pointer to a @ref enyx_hw_top.
     */
    enyx_hw_top const *
    handle() const noexcept;

private:
    enyx::hw::core subtree_root_;
    std::unique_ptr<enyx_hw_top> hw_top_c_;
};

} /* namespace hw_top */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/hw_top/hw_top.ipp>
