#pragma once

#include <chrono>
#include <memory>

#include <enyx/hw/core.hpp>
#include <enyx/cores/macros.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

#include <enyx/cores_c/flash/flash.h>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_flash>
{
    void
    operator()(::enyx_flash * ptr) const
    {
        ::enyx_flash_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace flash {

/// @copydoc ENYX_FLASH_SECTION_PROGRAM
static auto const SECTION_PROGRAM = ::ENYX_FLASH_SECTION_PROGRAM;

/// @copydoc ENYX_FLASH_SECTION_USER
static auto const SECTION_USER = ::ENYX_FLASH_SECTION_USER;

/// @copydoc ENYX_FLASH_SECTION_PARAMS
static auto const SECTION_PARAMS = ::ENYX_FLASH_SECTION_PARAMS;

/**
 * Main flash manipulation object.
 *
 * This wrapper provides @b RAII of @b C @ref enyx_flash
 */
class flash {
public:
    /**
     * The flash memory section representation.
     *
     * It is used to interact with a flash memory section (read / write / get
     * size).
     */
    class section {
    public:
        /**
         * Construct a flash section object from a @b C @ref enyx_flash_section
         * pointer.
         *
         * @param section Pointer to a @ref enyx_flash_section.
         *
         * @note This constructor should not be used directly, sections should
         * retrieved from @ref flash object section getters instead.
         */
        section(const enyx_flash_section * section);

        /**
         * Read data from a flash memory section.
         *
         * @param data The byte array in which data will be dumped.
         * @param offset The offset in byte from the beginning of the section
         * after which data will be read.
         * @param size The size of the data array in bytes.
         * @return A result object containing an error on failure.
         */
        enyx::result<void>
        read(uint8_t * data, size_t offset, size_t size) const noexcept;

        /**
         * Write data to a flash memory section.
         *
         * @param data The byte array containing data to write.
         * @param offset The offset in byte from the beginning of the section
         * after which data will be written.
         * @param size The size of the data array in bytes.
         * @return A result object containing an error on failure.
         */
        enyx::result<void>
        write(const uint8_t * data, size_t offset, size_t size) const noexcept;

        /**
         * Get the flash section start address.
         *
         * @return A result object containing the section start address on
         * success or an error on failure.
         */
        enyx::result<size_t>
        start_address() const noexcept;

        /**
         * Get the flash section size.
         *
         * @return A result object containing the section size on success or an
         * error on failure.
         */
        enyx::result<uint32_t>
        size() const noexcept;

        /**
         * Get the flash section name.
         *
         * @return A result object containing the section name on success or an
         * error on failure.
         */
        enyx::result<std::string>
        name() const noexcept;

    private:
        const ::enyx_flash_section * section_;
    };

    /**
     * Create the flash management object from a core subtree.
     *
     * @param core The root core of a subtree containing a flash core.
     * @throw system_error on failure
     */
    flash(enyx::hw::core const & core);

    /**
     * Read data from a flash controller register.
     *
     * @param reg_addr The flash register address.
     * @param buffer The byte array in which data will be dumped.
     * @param size The size of the data array in bytes.
     *
     * @return A result object containing an error on failure.
     */
    enyx::result<void>
    read_register(size_t reg_addr, uint8_t * buffer, size_t size) const noexcept;

    /**
     * Write data to a flash controller register.
     *
     * @param reg_addr The flash register address.
     * @param data The byte array containing data to write.
     * @param size The size of the data array in bytes.
     *
     * @return A result object containing an error on failure.
     */
    enyx::result<void>
    write_register(size_t reg_addr, uint8_t * data, size_t size) noexcept;

    /**
     * Set the FPGA to reload its firmware upon next reboot.
     *
     * @return A result object containing an error on failure.
     */
    enyx::result<void>
    set_fpga_reload_at_reboot() noexcept;

    /**
     * Reset the flash memory controller configuration.
     *
     * @return A result object containing an error on failure.
     */
    enyx::result<void>
    reset_memory() noexcept;

    /**
     * @deprecated This function has no effect
     *
     * Enable or disable backup of remaining data from erased sections during
     * memory write.
     *
     * @param enable Whether to enable (@b true) or disable (@b false) the
     * backup.
     * @return A result object containing an error on failure.
     */
    ENYX_CORES_CXX_DEPRECATED("this function has no effect")
    enyx::result<void>
    set_erase_backup(bool enable) noexcept;

    /**
     * Get the full flash memory size.
     *
     * @return A result object containing the memory size on success or an
     * error on failure.
     */
    enyx::result<size_t>
    get_memory_size() const noexcept;

    /**
     * Get the flash sector size.
     *
     * @return A result object containing the sector size on success or an
     * error on failure.
     */
    enyx::result<size_t>
    get_sector_size() const noexcept;

    /**
     * Check if firmware warm update is supported on this board.
     *
     * @return A result object containing the warm update support on success or
     * an error on failure.
     */
    enyx::result<bool>
    is_warm_update_supported() const noexcept;

    /**
     * Get flash memory sections by name.
     *
     * @param name The flash section name.
     * @return A result object containing the found sections on success (vector
     * can still be empty) or an error on failure.
     */
    enyx::result<std::vector<section>>
    get_sections(std::string const & name) const noexcept;

    /**
     * Get all flash memory sections.
     *
     * @return A result object containing the sections on success or an error
     * on failure.
     */
    enyx::result<std::vector<section>>
    get_sections() const noexcept;

    /**
     * @deprecated
     * Use @ref get_sections with SECTION_PROGRAM instead.
     *
     * Get the program section.
     *
     * @return A reference to the program section object.
     */
    ENYX_CORES_CXX_DEPRECATED("replaced by get_sections")
    flash::section const &
    pgm() const noexcept;

    /**
     * @deprecated
     * Use @ref get_sections with SECTION_USER instead.
     *
     * Get the user section.
     *
     * @return A reference to the user section object.
     */
    ENYX_CORES_CXX_DEPRECATED("replaced by get_sections")
    flash::section const &
    usr() const noexcept;

    /**
     * @deprecated
     * Use @ref get_sections with SECTION_PARAMS instead and take the first
     * returned section.
     *
     * Get the params section.
     *
     * @return A reference to the params section object.
     */
    ENYX_CORES_CXX_DEPRECATED("replaced by get_sections")
    flash::section const &
    prm() const noexcept;

    /**
     * @deprecated
     * Use @ref get_sections with SECTION_PARAMS instead and take the second
     * returned section.
     *
     * Get the OTP section.
     *
     * @return A reference to the OTP section object.
     */
    ENYX_CORES_CXX_DEPRECATED("replaced by get_sections")
    flash::section const &
    otp() const noexcept;

    /**
     * Access to the @b C handle.
     *
     * @return the @b C handle.
     */
    enyx_flash const *
    handle() const noexcept;

    /**
     * Access to the @b C handle.
     *
     * @return the @b C handle.
     */
    enyx_flash *
    handle() noexcept;

private:
    enyx::hw::core subtree_root_;
    std::unique_ptr<::enyx_flash> flash_;

    section pgm_;
    section usr_;
    section prm_;
    section otp_;
};

} /* namespace flash */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/flash/flash.ipp>
