#[=======================================================================[.rst:
EnyxHwGenFunctions
------------------

Enyx Hardware Generator Functions Definition
^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

.. command:: generate_hw_gen_targets([DESTINATION <destination>] [TEMPLATE_FOLDER <path>] [TEMPLATE <template_name>] INPUTS <files>...)

    DESTINATION
        Output folder for generated code. Default to `${PROJECT_BINARY_DIR}`.

    TARGET_PREFIX
        Prefix to add before the cmake target name for each core. The target
        format is `prefix_<core_name>`, or `<core_name>` if prefix is empty.
        Default to empty string.

    TEMPLATE_FOLDER
        Find templates in a directory. Default to /usr/share/enyx-hw/templates/

    TEMPLATE
        Template name to use. Default to `cpp/core.jinja`

    INPUTS
        Input xml core descriptions files.

   The ``generate_hw_gen_targets()`` generate the hw gen targets

   .. code-block:: cmake

       generate_hw_gen_targets(
           [DESTINATION <destination>]
           [TARGET_PREFIX <prefix>]
           [TEMPLATE_FOLDER <path>]
           [TEMPLATE <template_name>]
           INPUTS <files>...)
#]=======================================================================]
function(generate_hw_gen_targets)
    include(CMakeParseArguments)

    cmake_parse_arguments(
        ARG
        ""
        "DESTINATION;TARGET_PREFIX;TEMPLATE_FOLDER"
        "INPUTS;TEMPLATE"
        ${ARGN}
    )

    if(NOT ARG_DESTINATION)
        set(destination ${CMAKE_CURRENT_BINARY_DIR})
    else()
        set(destination ${ARG_DESTINATION})
    endif()

    if(ARG_TARGET_PREFIX)
        set(prefix "${ARG_TARGET_PREFIX}_")
    else()
        set(prefix "")
    endif()

    if(ARG_TEMPLATE_FOLDER)
        set(template_location ${ARG_TEMPLATE_FOLDER})
    else()
        set(template_location ${ENYX_HW_GEN_DEFAULT_TEMPLATE_FOLDER})
    endif()

    if(NOT ARG_TEMPLATE)
        set(template "cpp/core.jinja")
    else()
        set(template ${ARG_TEMPLATE})
    endif()

    find_program(ENYX_HW_CODEGEN_BINARY enyx-hw-codegen REQUIRED)

    foreach(xml_file ${ARG_INPUTS})
        get_filename_component(core_name ${xml_file} NAME_WE)
        set(output_dir ${core_name})
        set(target_name "${prefix}${core_name}")
        add_custom_command(
            OUTPUT
                ${destination}/${output_dir}/
            COMMAND
                ${ENYX_HW_CODEGEN_BINARY}
                    --output ${destination}/${output_dir}
                    --template-folder ${template_location}
                    --templates ${template}
                    --xml-descriptions ${xml_file}
            COMMENT
                "Generating ${template} for ${xml_file}"
        )
        add_custom_target(
            ${target_name}
            ALL
            DEPENDS
                ${destination}/${output_dir}/
                ${ENYX_HW_CODEGEN_BINARY}
        )
    endforeach()
endfunction()

get_filename_component(ENYX_HW_GEN_DEFAULT_TEMPLATE_FOLDER
                       ${CMAKE_CURRENT_LIST_DIR}/../templates
                       ABSOLUTE
                       CACHE)
