/*
 * HFP bus interface.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef HFP_BUS_H
#define HFP_BUS_H

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/semaphore.h>
#include <linux/completion.h>
#include <linux/cdev.h>
#include <linux/list.h>
#include <linux/io.h>

#include "enyx_hfp_irq_list.h"

/**
 *
 */
int
enyx_hfp_bus_type_register(void);

/**
 *
 */
void
enyx_hfp_bus_type_unregister(void);

/**
 *
 */
#define BUSNAMESIZE 128
struct enyx_hfp_bus {
    phys_addr_t phys_addr;
    uint8_t __iomem * virt_addr;
    struct enyx_hfp_irq_list * irq_list;
    unsigned int index;

    struct semaphore devices_lock;
    struct list_head devices;
    size_t devices_used;

    struct device device;
    struct module * owner;

    char bus_alias[BUSNAMESIZE];
};

/**
 *
 */
#define to_enyx_hfp_bus(n) \
    container_of(n, struct enyx_hfp_bus, device)

/**
 *
 */
#define create_enyx_hfp_bus(_addr, _irq_list, _parent)    \
    __create_enyx_hfp_bus(_addr, _irq_list, _parent,      \
                     THIS_MODULE)


struct enyx_hfp_bus *
__create_enyx_hfp_bus(phys_addr_t addr,
                 struct enyx_hfp_irq_list * irq_list,
                 struct device * parent,
                 struct module * owner);

/**
 *
 */
void
destroy_enyx_hfp_bus(struct enyx_hfp_bus * bus);

/**
 *
 */
int
clear_enyx_hfp_bus(struct enyx_hfp_bus * bus);

/**
 *
 */
int
scan_enyx_hfp_bus(struct enyx_hfp_bus * bus);

int
get_enyx_hfp_bus_version(struct enyx_hfp_bus * bus,
                         uint8_t * major,
                         uint8_t * minot,
                         uint8_t * patch);

/**
 *
 */
struct enyx_hfp_device_id {
    uint16_t hardware_id;
    uint8_t version_major;
    uint8_t version_minor;
    uint16_t revision;
};

/**
 *
 */
#define HFP_COMPUTE_VERSION(_major, _minor) \
    ((_major) << 8 | (_minor))


/**
 *
 */
struct enyx_hfp_device {
    struct enyx_hfp_device_id id;
    char modalias[128];
    uint64_t bus_offset;
    size_t size;
    char *driver_override;

    struct list_head bus_node;
    struct enyx_hfp_bus * bus;

    struct list_head node;
    struct list_head children;

    struct semaphore lock;

    struct device device;
};

#define HFP_REG_ADDR(index, offset) (4 * (index) + (offset))

/**
 *
 */
static inline void
enyx_hfp_write8(struct enyx_hfp_device * dev, size_t offset, uint8_t v)
{
    return writeb(v, dev->bus->virt_addr + dev->bus_offset + offset);
}

/**
 *
 */
static inline void
enyx_hfp_write16(struct enyx_hfp_device * dev, size_t offset, uint16_t v)
{
    return writew(v, dev->bus->virt_addr + dev->bus_offset + offset);
}

/**
 *
 */
static inline void
enyx_hfp_write32(struct enyx_hfp_device * dev, size_t offset, uint32_t v)
{
    return writel(v, dev->bus->virt_addr + dev->bus_offset + offset);
}

/**
 *
 */
static inline void
enyx_hfp_write64(struct enyx_hfp_device * dev, size_t offset, uint64_t v)
{
    return writeq(v, dev->bus->virt_addr + dev->bus_offset + offset);
}

/**
 *
 */
static inline uint8_t
enyx_hfp_read8(struct enyx_hfp_device * dev, size_t offset)
{
    return readb(dev->bus->virt_addr + dev->bus_offset + offset);
}

/**
 *
 */
static inline uint16_t
enyx_hfp_read16(struct enyx_hfp_device * dev, size_t offset)
{
    return readw(dev->bus->virt_addr + dev->bus_offset + offset);
}

/**
 *
 */
static inline uint32_t
enyx_hfp_read32(struct enyx_hfp_device * dev, size_t offset)
{
    return readl(dev->bus->virt_addr + dev->bus_offset + offset);
}

/**
 *
 */
static inline uint64_t
enyx_hfp_read64(struct enyx_hfp_device * dev, size_t offset)
{
    return readq(dev->bus->virt_addr + dev->bus_offset + offset);
}

/**
 *
 */
int
enyx_hfp_wait_until_val_is(struct enyx_hfp_device * dev, size_t offset, uint8_t val);

/**
 *
 */
size_t
enyx_hfp_read_str_unlocked(struct enyx_hfp_device * dev,
                      size_t str_reg,
                      size_t page_reg,
                      size_t ready_reg,
                      char * buffer, size_t buffer_capacity);

/**
 *
 */
size_t enyx_hfp_read32_locked_v0_2(struct enyx_hfp_device *dev,
        size_t read_reg,
        size_t page_reg,
        size_t ready_reg,
        uint16_t page_id);

/**
 *
 */
size_t enyx_hfp_read16_locked_v0_2(struct enyx_hfp_device *dev,
        size_t read_reg,
        size_t page_reg,
        size_t ready_reg,
        uint16_t page_id);

/**
 *
 */
void enyx_hfp_bus_for_each_safe(struct enyx_hfp_bus * bus,
                           void (*cb)(struct enyx_hfp_device *, void *),
                           void * opaque);

/**
 *
 */
static inline void *
enyx_hfp_get_drvdata(struct enyx_hfp_device *dev)
{
    return dev_get_drvdata(&dev->device);
}

/**
 *
 */
static inline void
enyx_hfp_set_drvdata(struct enyx_hfp_device *dev, void *data)
{
    dev_set_drvdata(&dev->device, data);
}

/**
 *
 */
static inline int
enyx_hfp_get_device_version(const struct enyx_hfp_device * dev)
{
    return HFP_COMPUTE_VERSION(dev->id.version_major,
                               dev->id.version_minor);
}

/**
 *
 */
struct enyx_hfp_device *
enyx_hfp_get_device_parent(struct enyx_hfp_device * root);

/**
 *
 */
int
enyx_hfp_for_each_child(struct enyx_hfp_device * device,
                   int (*fn)(struct enyx_hfp_device * dev, void * opaque),
                   void * opaque);

/**
 *
 */
inline static struct enyx_hfp_device *
enyx_hfp_device_get(struct enyx_hfp_device * dev)
{
    struct enyx_hfp_device * ret = NULL;
    struct enyx_hfp_bus * bus = dev->bus;

    /* The device driver module refcount is automatically incremented
     * when the device is opened but not the module
     * that created the bus instance (e.g. enyx_hfp_pcie).
     * Hence increment the bus count manually */
    if (! try_module_get(bus->owner))
        return ret;

    if (down_interruptible(&bus->devices_lock) < 0) {
        module_put(bus->owner);
        goto lock_failed;
    }

    /* Check if the bus has destoyed the current device while
     * it was waiting for the lock */
    if (list_empty(&bus->devices)) {
        module_put(bus->owner);
        goto bus_destroying;
    }

    ++ bus->devices_used;
    ret = dev;

bus_destroying:
    up(&bus->devices_lock);
lock_failed:
    return ret;
}

/**
 *
 */
inline static void
enyx_hfp_device_put(struct enyx_hfp_device * dev)
{
    down(&dev->bus->devices_lock);

    -- dev->bus->devices_used;

    up(&dev->bus->devices_lock);

    module_put(dev->bus->owner);
}

/**
 *
 */
struct enyx_hfp_device *
enyx_hfp_find_depth_first(struct enyx_hfp_device * device,
                     const struct enyx_hfp_device_id * id,
                     size_t device_index);

/**
 *
 */
struct enyx_hfp_driver {
    struct device_driver driver;

    const char * name;
    const struct enyx_hfp_device_id * id_table;

    int (*probe)(struct enyx_hfp_device * dev);
    void (*remove)(struct enyx_hfp_device * dev);
};

/**
 *
 */
#define to_enyx_hfp_driver(n) \
    container_of(n, struct enyx_hfp_driver, driver)

/**
 *
 */
#define DEFINE_HFP_DEVICE_TABLE(_table) \
    const struct enyx_hfp_device_id _table[]

/**
 *
 */
#define HFP_DEVICE_ANY_VERSION ((uint8_t)~0)

/**
 *
 */
#define HFP_DEVICE_ID_MAJOR_MINOR(_id, _major, _minor) \
    .hardware_id  = (_id), \
    .version_major = (_major), \
    .version_minor = (_minor),

/**
 *
 */
#define HFP_DEVICE_ID_MAJOR(_id, _major) \
    HFP_DEVICE_ID_MAJOR_MINOR(_id, _major, \
                              HFP_DEVICE_ANY_VERSION)

/**
 *
 */
#define HFP_DEVICE_ID(_id) \
    HFP_DEVICE_ID_MAJOR_MINOR(_id, \
                              HFP_DEVICE_ANY_VERSION, \
                              HFP_DEVICE_ANY_VERSION)

/**
 *
 */
#define MODULE_ALIAS_HFP_PREFIX "enyx_hfp:"

/**
 *
 */
#define MODULE_ALIAS_HFP_ID_MAJOR_MINOR(id, ma, mi) \
    MODULE_ALIAS(MODULE_ALIAS_HFP_PREFIX  #id "ma" #ma "mi" #mi)

/**
 *
 */
#define MODULE_ALIAS_HFP_ID_MAJOR(id, ma) \
    MODULE_ALIAS_HFP_ID_MAJOR_MINOR(id, ma, *)

/**
 *
 */
#define MODULE_ALIAS_HFP_ID(id) \
    MODULE_ALIAS_HFP_ID_MAJOR(id, *)

/**
 *
 */
#define enyx_hfp_register_driver(_driver) \
    __enyx_hfp_register_driver(_driver, THIS_MODULE, KBUILD_MODNAME)

int
__enyx_hfp_register_driver(struct enyx_hfp_driver * driver,
                      struct module * owner,
                      const char * owner_name);

/**
 *
 */
void
enyx_hfp_unregister_driver(struct enyx_hfp_driver * driver);

#define HFP_BUS_ADDR_SHIFT 2 // 32bits

#endif /* HFP_BUS_H */

