/**
 * @file
 * This file contains the c2a stream interface mock.
 *
 * It should be used to create custom c2a_stream mocks.
 */
#pragma once

#include <stdint.h>
#include <unistd.h>

#include <enyx/hw_c/c2a_stream.h>

/**
 * Represent a mocked C2A stream interface.
 *
 * In order to implement it, please set the corresponding callbacks and context
 * consistantly.
 */
typedef struct {
    /**
     * A user defined context to be passed to all callbacks.
     */
    void *
    context;

    /**
     * Called when the stream is created.
     *
     * @param context A user defined context to be used in the callback.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*on_create)(void * context);

    /**
     * Called when the stream is destroyed.
     *
     * @param context A user defined context to be used in the callback.
     */
    void
    (*on_destroy)(void * context);

    /**
     * Called to get the uid.
     *
     * @param context A user defined context to be used in the callback.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_uid)(void * context, char * buffer, size_t capacity);

    /**
     * Called to get the name.
     *
     * @param context A user defined context to be used in the callback.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_name)(void * context, char * buffer, size_t capacity);

    /**
     * Called to get the MTU.
     *
     * @param context A user defined context to be used in the callback
     * @param mtu The mtu to fill
     * @return 0 on success, -1 on error with @b errno set
     */
    int
    (*get_mtu)(void * context, uint32_t * mtu);

    /**
     * Called to get the size.
     *
     * @param context A user defined context to be used in the callback
     * @return the size on success, -1 on error with @b errno set
     */
    ssize_t
    (*get_size)(void * context);

    /**
     * Called to send @p size bytes of @p data over the stream.
     *
     * @param context A user defined context to be used in the callback.
     * @param data The data to send
     * @param size The size of the data to send.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*send)(void * context, void const * data, uint32_t size);

    /**
     * Called to send @p size bytes of @p data over the stream with @p flags.
     *
     * @param context A user defined context to be used in the callback.
     * @param data The data to send
     * @param size The size of the data to send.
     * @param flags The flags to set on data packet.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*send2)(void * context, void const * data, uint32_t size,
             enyx_hw_c2a_stream_flags flags);

    /**
     * Called to send @p vectors_count data @p vectors over the stream.
     *
     * @param context A user defined context to be used in the callback.
     * @param vectors The vectors to send.
     * @param vectors_count The number of vectors to send.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*send_io_vec)(void * context, enyx_hw_c2a_stream_io_vec const * vectors,
                   uint32_t vector_count);

    /**
     * Called to send @p vectors_count data @p vectors over the stream
     * with @p flags.
     *
     * @param context A user defined context to be used in the callback.
     * @param vectors The vectors to send.
     * @param vector_count The number of vectors to send.
     * @param flags The flags to set on data packets.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*send_io_vec2)(void * context, enyx_hw_c2a_stream_io_vec const *vectors,
                    uint32_t vector_count, enyx_hw_c2a_stream_flags flags);

    /**
     * Called to check if the stream  is ready.
     *
     * @param context A user defined context to be used in the callback.
     * @return true if the stream is ready, false otherwise.
     */
    bool
    (*is_ready)(void * context);

    /**
     * Called to check if the stream @p stream_index from accelerator
     * @p accelerator_index is idling (i.e. it is not currently processing
     * any data).
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return true if the stream is idling, false otherwise.
     */
    bool
    (*is_idle)(void * context);

    /**
     * Called to wait until the stream is idling (i.e. it is not currently
     * processing any data).
     *
     * @note This function should return -1 and set @b errno if the
     *       stream is still not idling after @p us_timeout.
     *
     * @param context A user defined context to be used in the callback.
     * @param us_timeout The duration of the wait.
     * @return true if the stream is idling, false otherwise.
     */
    int
    (*wait_until_idle)(void * context, uint32_t us_timeout);
} enyx_hwm_c2a_stream_interface;

/**
 * Create a mocked C2A stream interface.
 *
 * @note This function is a helper for higher level language bindings.
 *
 * @return The new mocked stream interface on success or NULL on failure (with
 * @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_c2a_stream_interface *
enyx_hwm_c2a_stream_interface_create();

/**
 * Destroy a mocked C2A stream interface.
 *
 * @note This function is a helper for higher level language bindings.
 *
 * @param stream The mocked stream interface to destroy
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hwm_c2a_stream_interface_destroy(enyx_hwm_c2a_stream_interface * stream);
