/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  cores description.
 */
#pragma once
#include <enyx/hw_c/symbol_visibility.h>
#include <enyx/hw_c/register_description.h>
#include <enyx/hw_c/core.h>

/**
 * This class represents a core description.
 *
 * It describes the registers it contains.
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_core_description enyx_hw_core_description;

/**
 * Create a new core description to be destroyed by
 * @ref enyx_hw_core_descr_destroy.
 *
 * This core description should be used to programatically define core
 * descriptions.
 *
 * @param name The core name
 * @param descriptor The descriptor (id/major/minor/revision) of the core.
 *
 * @return A new core description on success, NULL on failure
 *         (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_description *
enyx_hw_core_descr_new(char const * name,
                       enyx_hw_core_descriptor const * descriptor);

/**
 * Destroy a previously created core description.
 *
 * Only core descriptions created with @ref enyx_hw_core_descr_new
 * can be destroyed.
 *
 * @param description The description to be destroyed.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_core_descr_destroy(enyx_hw_core_description * description);

/**
 * Get the associated core name for a core description.
 *
 * @param description The core description to get info from
 * @param[out] buffer Output buffer for the name.
 * @param size The size of @p param
 *
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_descr_get_name(enyx_hw_core_description const * description,
                            char * buffer, size_t size);

/**
 * Get associated major/minor/revision/hardware id for a core description.
 *
 * @param description The core description to get info from
 * @param[out] descriptor The descriptor containing the new
 *                        major/minor/revision/id to be getted.
 * @return 0 on success, -1 if either @p description or @p descriptor does not
 *         exist (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_descr_get_descriptor(enyx_hw_core_description const * description,
                                  enyx_hw_core_descriptor * descriptor);

/**
 * Add a register description to a core description.
 *
 * Use this only for a core created with @ref enyx_hw_core_descr_new.
 *
 * @param description The core description to modify.
 * @param name The name of the new register.
 * @param reg The register description to add.
 * @return 0 on success, -1 if either @p description, @p name does or @p reg
 *         does not exist (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_descr_add_register_desc(enyx_hw_core_description * description,
                                     const char * name,
                                     enyx_hw_register_description const * reg);

/**
 * Find a register matching a name.
 *
 * @param description The core description to find the register from.
 * @param name The name of the register to find.
 * @param[out] reg The register to be found.
 * @return 0 on success, -1 if no such register was found or if either
 *         @p description, @p name does or @p reg does not exist (@b errno is
 *         set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_descr_find_register(enyx_hw_core_description const * description,
                                 const char * name,
                                 enyx_hw_register_description * reg);

/**
 * Get the number of registers for a core @p description.
 *
 * @param description The core description to search.
 * @return The number of registers on success, -1 on failure (with @b errno set
 *         accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_core_descr_count_registers(enyx_hw_core_description const * description);

/**
 * Get all registers of a core.
 *
 * @param description The core description to get the registers from.
 * @param regs Array of registers to be filled with @p description's registers
 * @param size Size of @p regs.
 * @return 0 on succes, -1 if size is too small or if either description or
 *         regs is NULL (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_descr_get_all_registers(enyx_hw_core_description const * description,
                                     enyx_hw_named_register * regs,
                                     size_t size);
