/**
 * @file
 *
 * Contains the types and functions related to the
 * Core discovery.
 */
#pragma once

#include <stdint.h>

#include <enyx/hw_c/mmio.h>
#include <enyx/hw_c/product.h>

/**
 * Represents an hw core identification properties
 *
 * @since 5.0.0
 */
typedef struct {
    /// The core major version
    uint8_t major;
    /// The core minor version
    uint8_t minor;
    /// The core revision
    uint16_t revision;
    /// The unique hardware id associated with this core
    uint16_t hardware_id;
} enyx_hw_core_descriptor;

/**
 * Represents an hw core
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_core enyx_hw_core;

/**
 * Represents a tree of hw core
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_core_tree enyx_hw_core_tree;

/**
 * Retrieve all cores within the @p channel address space
 *
 * @note This function will create individual shared mutexes for all cores in
 * @b /dev/shm with permissions @b 666. If multiple users of the library are
 * using the same machine, please make sure that they all have access to these
 * mutexes.
 * Additionally, if you are using a kernel >= 4.19, the shm will be subject to
 * an additional restriction (search protected_regular in @b man @b 5 @b proc.
 * It can be bypassed by discovering the cores for the first time as root or by
 * disabling it (@b protected_regular set to 0).
 *
 * @param channel The channel used to discover and access the cores
 * @return A core tree containing the whole core hierarchy
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_tree *
enyx_hw_enumerate_cores(enyx_hw_mmio * channel);

/**
 * Destroy a previously enumerated @p tree
 *
 * @param tree The tree to destroy
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_destroy_tree(enyx_hw_core_tree * tree);

/**
 * Get the root core of a core @p tree.
 *
 * @param tree The core tree
 * @return The root core on success, NULL on failure with errno set
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core *
enyx_hw_core_tree_get_root(enyx_hw_core_tree const * tree);

/**
 * Get the number of children for a @p core
 *
 * @param core The core to count the children of
 * @return the number of children for a core on success, -1 on error with errno
 *         set.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_core_count_children(enyx_hw_core const * core);

/**
 *  Retrieve the core child at @p index
 *
 *  @param parent The parent core
 *  @param index The child index
 *  @return The chid
 *
 *  @since 5.7.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core *
enyx_hw_core_get_child(enyx_hw_core const * parent,
                       size_t index);

/**
 * Get the children of a @p core.
 *
 * @param core The core to get the children of
 * @param[out] children Array containing the children if any.
 * @param size Capacity of @p children.
 * @return 0 on success, -1 with errno set on failure.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_get_children(enyx_hw_core const * core,
                          enyx_hw_core ** children,
                          size_t size);

/**
 * Get the parent core of a @p core.
 *
 * If the core has no parent, errno is set to ESRCH
 *
 * @param core The core we want to get the parent of
 * @return The parent node on success, NULL on error with errno set.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core *
enyx_hw_core_get_parent(enyx_hw_core const * core);

/**
 * Get the root core.
 *
 * @param core The core we want to get the root of
 * @return The root node on success, NULL on error with errno set.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core const *
enyx_hw_core_get_root(enyx_hw_core const * core);

/**
 * Get the base @p address of a @p core.
 *
 * @param core The core to get the base address of
 * @param[out] address The integer to fill with the address
 * @return 0 on success, -1 on error (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_get_base_address(enyx_hw_core const * core,
                              uint64_t * address);

/**
 * Get the address space size of a @p core
 *
 * @param core The core object to query
 * @return The address space size on succes, -1 on error
 *         (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_core_get_address_space_size(enyx_hw_core const * core);

/**
 * Get the descriptor of a @p core
 *
 * @param core The core object to query
 * @return The descriptor of the @p core on success, NULL on error
 *         (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_descriptor const *
enyx_hw_core_get_descriptor(enyx_hw_core const * core);

/**
 * Get the product version of a @p core.
 *
 * @note The product version is *not* the core version: the core version
 * defines compatibility with the MM API while the *product* version of a core
 * is arbitrary and is specific to each type of core. On cores which do not
 * have a *product* version, this function will fail with ENOTSUP.
 *
 * @param core The core object to query
 * @param[out] version The version to fill
 * @return 0 on success, -1 on error (@b errno is set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_get_product_version(enyx_hw_core const * core,
                                 enyx_hw_product_version * version);

/**
 * Get the mmio of a @p core
 *
 * @param core The core object to query
 * @return The mmio that contains @p core on success, NULL on error
 *         (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_mmio *
enyx_hw_core_get_mmio(enyx_hw_core const * core);

/**
 * Advisory lock a core.
 *
 * @param core The core to lock.
 * @return 0 on success, -1 if the core object does not exists
 *         (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_lock(enyx_hw_core * core);

/**
 * Advisory try to lock a core.
 *
 * @param core The core to lock.
 * @return True on succes, False on error (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_try_lock(enyx_hw_core * core);

/**
 * Unlock a core.
 *
 * @param core The core to unlock
 * @return 0 on success, -1 onf failure (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_unlock(enyx_hw_core * core);

/**
 * Hardware Reset a core
 *
 * @param core The core to reset
 * @return 0 on success, -1 onf failure (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_reset(enyx_hw_core *core);

/**
 * Find every descendant of @p ancestor that matches a given @p hardware_id.
 *
 * If @p capacity is too small to add some cores, they are counted but not
 * added to @p cores.
 *
 * @param ancestor An ancestor to the cores we are looking for.
 * @param hardware_id The hardware id we are looking for.
 * @param[out] cores An array that will contain the cores on success.
 * @param capacity The size of @p cores
 *
 * @return The total core count on success, -1 on failure (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_core_enumerate(enyx_hw_core const * ancestor,
                       uint16_t hardware_id,
                       enyx_hw_core ** cores,
                       size_t capacity);

