/**
 * @file
 * 
 * Contains the types used to retrieve optional results from various API calls.
 *
 * Copied from https://github.com/DavidKeller/kademlia/blob/master/src/kademlia/r.hpp
 */
#pragma once

#include <cassert>
#include <system_error>
#include <enyx/hw/namespace.hpp>

ENYX_HW_NAMESPACE_BEGIN

/**
 * @brief Designed to return a value <b>or</b> an error code.
 * @tparam ReturnType The value type to return if no error occured.
 *
 * @since 5.0.0
 */
template< typename ReturnType >
class result
{
public:
    /**
     * Exception type thrown when a value is accessed
     * while the result contains an error.
     *
     * @since 5.0.0
     */
    using exception_type = std::system_error;

    /**
     * The error type used to store error.
     *
     * @since 5.0.0
     */
    using error_type = std::error_code;

    /**
     * The value stored when no error occured.
     *
     * @since 5.0.0
     */
    using value_type = ReturnType;

public:
    /**
     * @brief Construct this initialized with
     *        a value whose emplace constructor
     *        arguments are provided
     *        to this constructor.
     * @tparam Args Value constructor arguments type.
     * @param args Value constructor arguments.
     *
     * @since 5.0.0
     */
    template< typename ...Args >
    result
        ( Args &&... args)
        : error_{ }
        , is_error_{ false }
    { construct_value( std::forward< Args >( args )... ); }

    /**
     * @brief Construct this initialized with an error.
     * @param new_error A constant reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( error_type const & new_error )
        : error_{ new_error }
        , is_error_{ true }
    { }

    /**
     * @brief Construct this initialized with an error.
     * @param new_error A mutable reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( error_type & new_error )
        : result{ const_cast< error_type const & >( new_error ) }
    { }

    /**
     * @brief Construct this initialized with an error.
     * @param new_error A rvalue reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( error_type && new_error )
        : result{ const_cast< error_type const & >( new_error ) }
    { }

    /**
     * @brief Copy constructor.
     * @param other A constant reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( result const & other )
        : error_{ other.error_ }
        , is_error_{ other.is_error_ }
    { if ( other ) construct_value( other.v() ); }

    /**
     * @brief Copy constructor.
     * @param other A mutable reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( result & other )
        : result{ const_cast< result const & >( other ) }
    { }

    /**
     * @brief Copy constructor.
     * @param other A rvalue reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( result && other )
        : error_{ std::move( other.error_ ) }
        , is_error_{ other.is_error_ }
    { if ( other ) construct_value( std::move( other.v() ) ); }

    /**
     * @brief Destructor.
     *
     * @since 5.0.0
     */
    ~result
        ( void )
    { destruct_value_if_present(); }

    /**
     * @brief Assignment operator.
     * @param other A constant reference to an error.
     *
     * @since 5.0.0
     */
    result &
    operator=
        ( result const & other )
    { *this = result{ other }; }

    /**
     * @brief Assignment operator.
     * @param other A rvalue reference to an error.
     *
     * @since 5.0.0
     */
    result &
    operator=
        ( result && other )
    {
        destruct_value_if_present();
        error_ = other.error_;
        is_error_ = other.is_error_;

        if ( other ) construct_value( std::move( other.v() ) );

        return *this;
    }

    /**
     * @brief Assignment from an error operator.
     * @param new_error The error to assign to this.
     * @return A reference to this.
     *
     * @since 5.0.0
     */
    result &
    operator=
        ( error_type const & new_error )
    {
        assert( new_error && "unexpected success error code" );
        destruct_value_if_present();
        error_ = new_error;
        is_error_ = true;

        return *this;
    }

    /**
     * @brief Assignment from a value operator.
     * @param new_value The value to assign to this.
     * @return A reference to this.
     *
     * @since 5.0.0
     */
    result &
    operator=
        ( value_type const & new_value )
    {
        destruct_value_if_present();
        error_.clear();
        construct_value( new_value );
        is_error_ = false;

        return *this;
    }

    /**
     * @brief Assignment from a value operator.
     * @param new_value The value to assign to this.
     * @return A reference to this.
     *
     * @since 5.0.0
     */
    result &
    operator=
        ( value_type && new_value )
    {
        destruct_value_if_present();
        error_.clear();
        construct_value( std::move( new_value ) );
        is_error_ = false;

        return *this;
    }

    /**
     * @brief bool operator used to check if this
     *        has been initialized with a value.
     * @return true if initliazed from a value,
     *         false otherwise.
     *
     * @since 5.0.0
     */
    explicit
    operator bool() const noexcept
    { return ! is_error_; }

    /**
     * @brief Get the value stored in this.
     * @return A reference to the stored value.
     * @note throw exception_type if this is initiliazed with an error.
     * @throw exception_type
     *
     * @since 5.0.0
     */
    value_type &
    v() &
    { return access_value_or_throw(); }

    /**
     * @copydoc value_type & v()
     *
     * @since 5.0.0
     */
    value_type const &
    v() const &
    { return access_value_or_throw(); }

    /**
     * @brief Get the value stored in this.
     * @return A r-value reference to the stored value.
     * @note throw exception_type if this is initiliazed with an error.
     * @throw exception_type
     *
     * @since 5.0.0
     */
    value_type &&
    v() &&
    { return std::forward<value_type>(access_value_or_throw()); }

    /**
     * @copydoc value_type & v()
     *
     * @since 5.0.0
     */
    value_type const &&
    v() const &&
    { return std::forward<value_type>(access_value_or_throw()); }

    /**
     * @brief Get the error stored in this.
     * @return A constant reference to the stored error.
     *
     * @since 5.0.0
     */
    error_type const &
    e() const
    { return error_; }

    /**
     * @brief Get the value stored in this.
     * @return A reference to the stored value.
     * @note throw exception_type if this is initiliazed with an error.
     * @throw exception_type
     *
     * @since 5.0.0
     */
    value_type &
    value() &
    { return access_value_or_throw(); }

    /**
     * @copydoc value_type & v()
     *
     * @since 5.0.0
     */
    value_type const &
    value() const &
    { return access_value_or_throw(); }

    /**
     * @brief Get the value stored in this.
     * @return A r-value reference to the stored value.
     * @note throw exception_type if this is initiliazed with an error.
     * @throw exception_type
     *
     * @since 5.0.0
     */
    value_type &&
    value() &&
    { return std::forward<value_type>(access_value_or_throw()); }

    /**
     * @copydoc value_type && v()
     *
     * @since 5.0.0
     */
    value_type const &&
    value() const &&
    { return std::forward<value_type>(access_value_or_throw()); }

    /**
     * @brief Get the error stored in this.
     * @return A constant reference to the stored error.
     *
     * @since 5.0.0
     */
    error_type const &
    error() const
    { return error_; }

private:
    /**
     * @brief Construct the value in the initilized value_ buffer.
     *
     * @since 5.0.0
     */
    template< typename ...Args >
    void
    construct_value
        ( Args &&... args )
    { ::new(&value_) value_type( std::forward< Args >( args )... ); }

    /**
     * @brief If a value has been stored in the value_ buffer,
     *        call the destructor.
     *
     * @since 5.0.0
     */
    void
    destruct_value_if_present()
    { if (*this) access_value_or_throw().~ReturnType(); }

    /**
     * @brief Return the value_ buffer as a value.
     *
     * @since 5.0.0
     */
    value_type &
    access_value_or_throw() &
    {
        if (! *this)
            throw exception_type{ error_ };

        return reinterpret_cast< value_type & >( value_ );
    }

    /**
     * @copydoc value_type & access_value_or_throw() &
     */
    value_type const &
    access_value_or_throw() const &
    {
        if (! *this)
            throw exception_type{ error_ };

        return reinterpret_cast< value_type const & >( value_ );
    }

    /**
     * @brief Return the value_ buffer as a r-value reference.
     *
     * @since 5.0.0
     */
    value_type &&
    access_value_or_throw() &&
    {
        if (! *this)
            throw exception_type{ error_ };

        return std::forward<value_type>(
                reinterpret_cast< value_type & >( value_ )
        );
    }

    /**
     * @copydoc value_type && access_value_or_throw() &&
     */
    value_type const &&
    access_value_or_throw() const &&
    {
        if (! *this)
            throw exception_type{ error_ };

        return std::forward<value_type>(
                reinterpret_cast< value_type const & >( value_ )
        );
    }


private:
    /* The error stored if no value has be set. */
    error_type error_;

    /*
     * A buffer used to store a value if this is not
     * initliazed with an error.
     */
    typename std::aligned_storage< sizeof( value_type )
                                 , std::alignment_of< value_type >::value
                                 >::type value_;
    bool is_error_ = false;
};

/**
 * @brief Designed to return nothing <b>or</b> an error code.
 *
 * @since 5.0.0
 */
template<>
class result<void>
{
public:
    /**
     * Exception type thrown when a value is accessed
     * while the result contains an error.
     *
     * @since 5.0.0
     */
    using exception_type = std::system_error;

    /**
     * The error type used to store error.
     *
     * @since 5.0.0
     */
    using error_type = std::error_code;

    /**
     * The value stored when no error occured.
     *
     * @since 5.0.0
     */
    using value_type = void;

public:
    /**
     * @brief Construct this initialized with
     *        a value whose emplace constructor
     *        arguments are provided
     *        to this constructor.
     * @tparam Args Value constructor arguments type.
     *
     * @since 5.0.0
     */
    template< typename ...Args >
    result
        ()
        : error_{ }
        , is_error_{ false }
    {}

    /**
     * @brief Construct this initialized with an error.
     * @param new_error A constant reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( error_type const & new_error )
        : error_{ new_error }
        , is_error_{ true }
    { }

    /**
     * @brief Construct this initialized with an error.
     * @param new_error A mutable reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( error_type & new_error )
        : result{ const_cast< error_type const & >( new_error ) }
    { }

    /**
     * @brief Construct this initialized with an error.
     * @param new_error A rvalue reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( error_type && new_error )
        : result{ const_cast< error_type const & >( new_error ) }
    { }

    /**
     * @brief Copy constructor.
     * @param other A constant reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( result const & other )
        : error_{ other.error_ }
        , is_error_{ other.is_error_ }
    { }

    /**
     * @brief Copy constructor.
     * @param other A mutable reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( result & other )
        : result{ const_cast< result const & >( other ) }
    { }

    /**
     * @brief Copy constructor.
     * @param other A rvalue reference to an error.
     *
     * @since 5.0.0
     */
    result
        ( result && other )
        : error_{ std::move( other.error_ ) }
        , is_error_{ other.is_error_ }
    { }

    /**
     * @brief Destructor.
     */
    ~result
        ( void )
    {  }

    /**
     * @brief Move assignment operator.
     * @param other Object to move from.
     * @return A reference to this.
     *
     * @since 5.0.0
     */
    result &
    operator=
        ( result && other )
    {
        error_ = std::move(other.error_);
        is_error_ = other.is_error_;
        return *this;
    }

    /**
     * @brief Assignment from an error operator.
     * @param new_error The error to assign to this.
     * @return A reference to this.
     *
     * @since 5.0.0
     */
    result &
    operator=
        ( error_type const & new_error )
    {
        assert( new_error && "unexpected success error code" );
        error_ = new_error;
        is_error_ = true;
        return *this;
    }

    /**
     * @brief bool operator used to check if this
     *        has been initialized with a value.
     * @return true if initliazed from a value,
     *         false otherwise.
     *
     * @since 5.0.0
     */
    explicit
    operator bool() const noexcept
    { return ! is_error_; }

    /**
     * @brief Get the value stored in this.
     * @return A reference to the stored value.
     * @note throw exception_type if this is initiliazed with an error.
     * @throw exception_type
     *
     * @since 5.0.0
     */
    value_type
    v()
    { return access_value_or_throw(); }

    /**
     * @copydoc value_type & v()
     *
     * @since 5.0.0
     */
    value_type
    v() const
    { return access_value_or_throw(); }

    /**
     * @brief Get the error stored in this.
     * @return A constant reference to the stored error.
     *
     * @since 5.0.0
     */
    error_type const &
    e() const
    { return error_; }

    /**
     * @brief Get the value stored in this.
     * @return A reference to the stored value.
     * @note throw exception_type if this is initiliazed with an error.
     * @throw exception_type
     *
     * @since 5.0.0
     */
    value_type
    value()
    { return access_value_or_throw(); }

    /**
     * @copydoc value_type & v()
     *
     * @since 5.0.0
     */
    value_type
    value() const
    { return access_value_or_throw(); }

    /**
     * @brief Get the error stored in this.
     * @return A constant reference to the stored error.
     *
     * @since 5.0.0
     */
    error_type const &
    error() const
    { return error_; }

private:
    /**
     * @brief Return the value_ buffer as a value.
     *
     * @since 5.0.0
     */
    value_type
    access_value_or_throw()
    {
        if (! *this)
            throw exception_type{ error_ };

        return;
    }

    /**
     * @copydoc value_type & access_value_or_throw()
     */
    value_type
    access_value_or_throw() const
    {
        if (! *this)
            throw exception_type{ error_ };

        return;
    }


private:
    /* The error stored if no value has be set. */
    error_type error_;
    bool is_error_;
};

ENYX_HW_NAMESPACE_END
