/**
 * @file
 *
 * Contains the basic "mocking" interface.
 */
#pragma once

#include <cstdint>

#include <list>
#include <vector>
#include <exception>
#include <functional>
#include <cmath>
#include <cstring>
#include <limits>

#include <enyx/hw/namespace.hpp>
#include <enyx/hw_c/mock.h>
#include <enyx/hw/macros.hpp>

ENYX_HW_NAMESPACE_BEGIN
namespace mocking {

///@cond
/// @copydoc enyx_hw_mock
class ENYX_HW_CXX_DEPRECATED("use the ::enyx_hw_mock structure directly instead") mock {
public:
    /**
     * Constructor
     */
    mock() noexcept;

    /**
     * Deleted copy constructor
     */
    mock(mock const &) = delete;

    /**
     * Deleted copy assignment operator
     */
    mock operator=(mock const &) = delete;

    /**
     * Move constructor
     */
    mock(mock && other);

    /**
     * Move assignment operator
     */
    mock& operator=(mock && other);

    /**
     * Access the C handle.
     *
     * @return The C handle.
     */
    enyx_hw_mock const &
    handle() const noexcept;

    /**
     * Access the C handle.
     *
     * @return The C handle.
     */
    enyx_hw_mock &
    handle() noexcept;

    /**
     * Called to retrieve the count of accelerators
     *
     * @return the number of detected accelerators.
     */
    std::function<size_t(void)> count_accelerators;

    /**
     * Called to copy uid of accelerator at @p accelerator_index
     * into @p buffer.
     *
     * @param accelerator_index The accelerator index.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      char * buffer,
                      size_t capacity)> get_accelerator_uid;

    /**
     * Called to copy name of accelerator at @p accelerator_index
     * into @p buffer.
     *
     * @param accelerator_index The accelerator index.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      char * buffer, size_t capacity)> get_accelerator_name;

    /**
     * Called to retrieve the count of mmios belonging to
     * accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @return The number of mmios on success, 0 with errno set on error.
     */
    std::function<size_t(uint32_t accelerator_index)> count_mmios;

    /**
     * Called to retrieve the size of the mmio address space belonging
     * to accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param size The size integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      size_t * size)> get_mmio_size;

    /**
     * Called to retrieve the uid of the mmio belonging to
     * accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      char * buffer, size_t capacity)> get_mmio_uid;

    /**
     * Called to retrieve the name of the mmio belonging to
     * accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      char * buffer, size_t capacity)> get_mmio_name;

    /**
     * Called to write the @p value to address @p addr within
     * mmio address space.
     *
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param addr The address of the write (in bytes).
     * @param value The integral value to write.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      uint64_t addr, uint8_t value)> mmio_write_8;

    /**
     * @copydoc mmio_write_8
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      uint64_t addr, uint16_t value)> mmio_write_16;

    /**
     * @copydoc mmio_write_8
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      uint64_t addr, uint32_t value)> mmio_write_32;

    /**
     * @copydoc mmio_write_8
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      uint64_t addr, uint64_t value)> mmio_write_64;

    /**
     * Called to read the @p value at address @p addr within
     * mmio address space.
     *
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param addr The address of the read (in bytes).
     * @param value The integer to fill
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      uint64_t addr, uint8_t * value)> mmio_read_8;

    /**
     * @copydoc mmio_read_8
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      uint64_t addr, uint16_t * value)> mmio_read_16;

    /**
     * @copydoc mmio_read_8
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      uint64_t addr, uint32_t * value)> mmio_read_32;

    /**
     * @copydoc mmio_read_8
     */
    std::function<int(uint32_t accelerator_index, uint32_t mmio_index,
                      uint64_t addr, uint64_t * value)> mmio_read_64;

    /**
     * Called to retrieve the number of c2a streams belonging to
     * accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @return the number of detected c2a streams accelerators.
     */
    std::function<size_t(uint32_t accelerator_index)> count_c2a_streams;

    /**
     * Called when the stream at @p stream_index belonging accelerator @p
     * accelerator_index is created.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index)>
        on_c2a_stream_create;

    /**
     * Called when the stream at @p stream_index belonging accelerator @p
     * accelerator_index is destroyed.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     */
    std::function<void(uint32_t accelerator_index, uint32_t stream_index)>
        on_c2a_stream_destroy;

    /**
     * Called to copy the uid of c2a stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index,
                      char * buffer, size_t capacity)> get_c2a_stream_uid;

    /**
     * Called to copy the name of c2a stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index,
                      char * buffer, size_t capacity)> get_c2a_stream_name;

    /**
     * Called to copy the mtu of c2a stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p mtu.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param mtu The mtu to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index,
                      uint32_t *mtu)> get_c2a_stream_mtu;

    /**
     * Called to copy the size of c2a stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p size.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param size The size to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index,
                      size_t *size)> get_c2a_stream_size;

    /**
     * Called to send @p size bytes of @p data over the stream @p stream_index
     * from accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param data The data to send
     * @param size The size of the data to send.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      void const *data,
                      uint32_t size)> c2a_stream_send;

    /**
     * Called to send @p vectors_count data @p vectors over the stream
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param data The data to send
     * @param size The size of the data to send.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      enyx_hw_c2a_stream_io_vec const *vectors,
                      uint32_t vectors_count)> c2a_stream_send_io_vec;

    /**
     * Called to check if the stream @p stream_index from accelerator
     * @p accelerator_index is ready.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return true if the stream is ready, false otherwise.
     */
    std::function<bool(uint32_t accelerator_index,
                       uint32_t stream_index)> is_c2a_stream_ready;

    /**
     * Called to check if the stream @p stream_index from accelerator
     * @p accelerator_index is idling (i.e. it is not currently processing
     * any data).
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return true if the stream is idling, false otherwise.
     */
    std::function<bool(uint32_t accelerator_index,
                       uint32_t stream_index)> is_c2a_stream_idle;

    /**
     * Called to wait until the stream @p stream_index from accelerator
     * @p accelerator_index is idling (i.e. it is not currently processing
     * any data).
     *
     * @note This function should return -1 and set @b errno if the
     *       stream is still not idling after @p us_timeout.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param us_timeout The duration of the wait.
     * @return 0 if the stream is idle, -1 otherwise.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      uint32_t us_timeout)> wait_until_c2a_stream_is_idle;

    /**
     * Called to retrieve the number of a2c streams belonging to
     * accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @return the number of detected a2c streams accelerators.
     */
    std::function<size_t(uint32_t accelerator_index)> count_a2c_streams;

    /**
     * Called when the stream at @p stream_index belonging accelerator @p
     * accelerator_index is created.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index)>
        on_a2c_stream_create;

    /**
     * Called when the stream at @p stream_index belonging accelerator @p
     * accelerator_index is destroyed.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     */
    std::function<void(uint32_t accelerator_index, uint32_t stream_index)>
        on_a2c_stream_destroy;

    /**
     * Called to copy the uid of a2c stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index,
                      char * buffer, size_t capacity)> get_a2c_stream_uid;

    /**
     * Called to copy the name of a2c stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index,
                      char * buffer, size_t capacity)> get_a2c_stream_name;

    /**
     * Called to copy the mtu of a2c stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p mtu.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param mtu The mtu to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index,
                      uint32_t *mtu)> get_a2c_stream_mtu;

    /**
     * Called to copy the size of a2c stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p size.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param size The size to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index, uint32_t stream_index,
                      size_t *size)> get_a2c_stream_size;

    /**
     * Called to invoke @p on_data handler on received packet(s) of stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @note The function should return 0 if no data is available
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param on_data The handler to invoke on received packet.
     * @param opaque The opaque pointer to forward to the on_data handler.
     * @return The count of received packets.
     */
    std::function<size_t(uint32_t accelerator_index, uint32_t stream_index,
                         enyx_hw_a2c_stream_handler on_data,
                         void * opaque)> poll_a2c_stream_once;

    /**
     * Called to retrieve the total packet count received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param packet_count The packet count integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      uint32_t * packet_count)> get_a2c_stream_packet_count;

    /**
     * Called to retrieve the stream at @p stream_index from
     * accelerator @p accelerator_index backpressure.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param backpressure The backpressure integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      uint32_t * backpressure)> get_a2c_stream_backpressure;

    /**
     * Called to retrieve the total fifo errors received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param fifo_errors The fifo errors integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      uint32_t * fifo_errors)> get_a2c_stream_fifo_errors;

    /**
     * Called to retrieve the total truncated_count received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param truncated_count The truncated_count integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      uint32_t * truncated_count)> get_a2c_stream_truncated_count;

    /**
     * Called to retrieve the total errors received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param errors The errors integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      uint32_t * errors)> get_a2c_stream_errors;

    /**
     * Called to retrieve the total usage received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param usage The usage integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint32_t accelerator_index,
                      uint32_t stream_index,
                      uint32_t * usage)> get_a2c_stream_usage;

private:
    enyx_hw_mock mock_;
};
///@endcond

} /* namespace mocking */
ENYX_HW_NAMESPACE_END

#include <enyx/hw/mocking/mock.ipp>
