#pragma once

#include <cstdint>
#include <cstring>

#include <list>
#include <vector>
#include <sstream>
#include <exception>
#include <functional>
#include <cmath>
#include <cstring>
#include <limits>

#include <enyx/hw/core.hpp>
#include <enyx/hw/namespace.hpp>
#include <enyx/hw/register_description.hpp>

#include <enyx/hw/mocking/core.hpp>
#include <enyx/hw/mocking/register.hpp>

#include <enyx/hw_c/mock.h>
#include <enyx/hw_c/mocking/mmio.h>


///@cond
namespace std {
template<>
struct default_delete<::enyx_hwm_mmio>
{
    void
    operator()(::enyx_hwm_mmio * ptr) const
    {
        ::enyx_hwm_mmio_destroy(ptr);
    }
};
}
///@endcond

ENYX_HW_NAMESPACE_BEGIN
namespace mocking {

/**
 * This class represents a mocked MMIO device interface.
 *
 * In order to customize it, please inherit and register it using the mocked
 * accelerator class.
 *
 * You may likely want to use the complete @ref mmio implementation that is
 * able to mock cores and registers.
 */
class mmio_interface
{
public:
    mmio_interface();
    /**
     * Destructor.
     */
    virtual ~mmio_interface() = default;

    /**
     * Called to retrieve the size of the mmio address space.
     *
     * @return The MMIO size
     */
    std::function<size_t()> get_size;

    /**
     * Called to retrieve the uid of the mmio.
     *
     * @return The mmio uid
     */
    std::function<std::string()> get_uid;

    /**
     * Called to retrieve the name of the mmio.
     *
     * @return The mmio name.
     */
    std::function<std::string()> get_name;

    /**
     * Called to write the @p value to address @p addr within
     * mmio address space.
     *
     * @param addr The address of the write (in bytes).
     * @param value The integral value to write.
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint64_t addr, uint8_t value)> write_8;

    /**
     * @copydoc write_8
     */
    std::function<int(uint64_t addr, uint16_t value)> write_16;

    /**
     * @copydoc write_8
     */
    std::function<int(uint64_t addr, uint32_t value)> write_32;

    /**
     * @copydoc write_8
     */
    std::function<int(uint64_t addr, uint64_t value)> write_64;

    /**
     * Called to read the @p value at address @p addr within
     * mmio address space.
     *
     * @param addr The address of the read (in bytes).
     * @param value The integer to fill
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int(uint64_t addr, uint8_t * value)> read_8;

    /**
     * @copydoc read_8
     */
    std::function<int(uint64_t addr, uint16_t * value)> read_16;

    /**
     * @copydoc read_8
     */
    std::function<int(uint64_t addr, uint32_t * value)> read_32;

    /**
     * @copydoc read_8
     */
    std::function<int(uint64_t addr, uint64_t * value)> read_64;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_mmio_interface *
    handle() noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_mmio_interface const *
    handle() const noexcept;
private:
    std::unique_ptr<enyx_hwm_mmio_interface> mmio_;
};

using basic_mmio
    ENYX_HW_CXX_DEPRECATED("Replaced by mmio_interface") = mmio_interface;

/**
 * This class is used to bind the mock mmio callbacks to an instantiated
 * @ref core tree.
 */
class mmio : public mmio_interface
{
public:
    /**
     * Patch the @p mock to register mmio methods
     * that redirect access to @p root
     *
     * @param root The root mocked core to use
     * @param width The mmio address width used
     * @throws std::system_error if address @p width it is too small.
     */
    mmio(core & root, std::uint8_t width = 32);

    /**
     * Destructor.
     */
    virtual ~mmio();

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_mmio *
    mmio_handle() noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_mmio const *
    mmio_handle() const noexcept;
private:
    std::unique_ptr<::enyx_hwm_mmio> handle_;
};

} /* namespace mocking */
ENYX_HW_NAMESPACE_END

#include <enyx/hw/mocking/mmio.ipp>
