/**
 * @file
 * This file contains the class that can be used to mock a whole
 * accelerator.
 */
#pragma once
#include <vector>

#include <enyx/hw/mocking/a2c_stream.hpp>
#include <enyx/hw/mocking/c2a_stream.hpp>
#include <enyx/hw/mocking/mmio.hpp>
#include <enyx/hw/mocking/mock.hpp>
#include <enyx/hw/product.hpp>

#include <enyx/hw_c/mocking/accelerator.h>

///@cond
namespace std {

template<>
struct default_delete<::enyx_hwm_accelerator>
{
    void
    operator()(::enyx_hwm_accelerator * ptr) const
    {
        ::enyx_hwm_accelerator_destroy(ptr);
    }
};
}
///@endcond

ENYX_HW_NAMESPACE_BEGIN
namespace mocking {

/**
 * This class represents a mocked Accelerator. It can be used to fake a whole
 * accelerator.
 */
class accelerator
{
    using a2c_stream_ptr = std::unique_ptr<mocking::a2c_stream_interface>;
    using c2a_stream_ptr = std::unique_ptr<mocking::c2a_stream_interface>;
    using mmio_ptr = std::unique_ptr<mocking::mmio_interface>;
    using reset_cb_t = std::function<void()>;
public:
    ///@cond
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
    accelerator(mock & mock);
#pragma GCC diagnostic pop
    ///@endcond

    /**
     * Install an accelerator on a mock
     *
     * Set the corresponding callbacks on the mock.
     *
     * @param mock The mock to use
     */
    accelerator(::enyx_hw_mock * mock);

    /**
     * Install an accelerator on a mock
     *
     * Set the corresponding callbacks on the mock.
     *
     * @param mock The mock to use
     */
    accelerator(::enyx_hw_mock & mock);

    /**
     * Destructor.
     */
    ~accelerator();

    /**
     * Set the @p version of this accelerator.
     *
     * @param version The version to set
     * @return A result object containing an error on failure
     */
    result<void>
    set_version(product_version const & version) noexcept;

    /**
     * Set the @p name of this accelerator.
     *
     * @param name The name to set
     * @return A result object containing an error on failure
     */
    result<void>
    set_name(std::string const & name) noexcept;

    /**
     * Set the user count of a mocked accelerator.
     *
     * @param count The count to set
     * @return A result object containing an error on failure
     */
    result<void>
    set_user_count(size_t count) noexcept;

    /**
     * Set the reset callback of a mocked accelerator.
     *
     * @param cb The callback to set
     * @return A result object containing an error on failure
     */
    result<void>
    set_reset_callback(reset_cb_t && cb) noexcept;

    /**
     * Register an MMIO to this accelerator.
     *
     * @tparam MMIO The type of the MMIO
     * @tparam Args The types of MMIO's constructor arguments
     *
     * @param args Arguments of MMIO's constructor
     * @return A reference to the new MMIO
     */
    template<typename MMIO = mocking::mmio_interface, typename... Args>
    MMIO & add_mmio(Args&&... args);

    /**
     * Register an A2C stream to this accelerator.
     *
     * @tparam A2CStream The type of the A2C stream
     * @tparam Args The types of A2C stream's constructor arguments
     *
     * @param args Arguments of A2C stream's constructor
     * @return A reference to the new A2C stream
     */
    template<typename A2CStream = mocking::a2c_stream_interface, typename... Args>
    A2CStream & add_a2c_stream(Args&&... args);

    /**
     * Register an C2A stream to this accelerator.
     *
     * @tparam C2AStream The type of the C2A stream
     * @tparam Args The types of C2A stream's constructor arguments
     *
     * @param args Arguments of C2A stream's constructor
     * @return A reference to the new C2A stream
     */
    template<typename C2AStream = mocking::c2a_stream_interface, typename... Args>
    C2AStream & add_c2a_stream(Args&&... args);

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_accelerator *
    handle() noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_accelerator const *
    handle() const noexcept;

private:
    ::enyx_hw_mock * mock_;
    std::unique_ptr<::enyx_hwm_accelerator> handle_;
    std::vector<mmio_ptr> mmios_;
    std::vector<a2c_stream_ptr> a2c_streams_;
    std::vector<c2a_stream_ptr> c2a_streams_;
};

} /* namespace mocking */
ENYX_HW_NAMESPACE_END

#include <enyx/hw/mocking/accelerator.ipp>
