/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  core_descriptions.
 */
#pragma once

#include <enyx/hw_c/core_descriptions.h>

#include <string>
#include <system_error>
#include <enyx/hw/namespace.hpp>
#include <enyx/hw/core.hpp>
#include <enyx/hw/core_description.hpp>


ENYX_HW_NAMESPACE_BEGIN

/**
 * @copydoc enyx_hw_core_descriptions
 */
class core_descriptions
{
public:
    /**
     * Construct a core descriptions.
     *
     * @since 5.0.0
     */
    core_descriptions() noexcept;

    /**
     * Construct a core descriptions from an existing C core descriptions
     * (wrapped in a shared_ptr).
     *
     * @param descriptions The core descriptions to use.
     *
     * @since 5.0.0
     */
    core_descriptions(std::shared_ptr<enyx_hw_core_descriptions> const & descriptions) noexcept;

    /**
     * Add assignment operator (Union).
     *
     * A failure can happen if both core_descriptions have the same hardware_id
     * for two cores with different names (unlikely).
     *
     * @return @b this, updated with @p rhs.
     * @throws std::system_error if @p rhs is not consistent with @b this.
     *
     * @since 5.0.0
     */
    core_descriptions operator+=(const core_descriptions & rhs);

    /**
     * Add a core description to @b this
     *
     * @param core The core description to add.
     * @return A result object that can be used to get an error (if any).
     *
     * @since 5.0.0
     */
    result<void>
    add_core_description(core_description const& core) noexcept;

    /**
     * Find the best core_description for the core @p descriptor.
     *
     * @param descriptor The core descriptor to match
     *
     * @return A result object containing either a core_description or an
     *         error.
     *
     * @since 5.0.0
     */
    result<core_description>
    find(core::descriptor_type const& descriptor) const noexcept;

    /**
     * Find the best core_description for the core @p core.
     *
     * @param core The core requesting a description
     *
     * @return A result object containing either a core_description or an
     *         error.
     *
     * @since 5.0.0
     */
    result<core_description>
    find(core const& core) const noexcept;

    /**
     * Find The @b hardware_id corresponding to a given core @p name.
     *
     * @param name The name of the core being looked up.
     * @return A result object containing either the @p hardware_id or an
     *         error if it was not found.
     *
     * @since 5.0.0
     */
    result<std::uint16_t>
    get_hardware_id(std::string const & name) const noexcept;

    /**
     * Find The @b name corresponding to a given core @p hardware_id.
     *
     * @param hardware_id The hardware_id of the core being looked up.
     * @return A result object containing either the @p name or an
     *         error if it was not found.
     *
     * @since 5.0.0
     */
    result<std::string>
    get_name(std::uint16_t hardware_id) const noexcept;

    /**
     * Direct access to the underlying C enyx_hw_core_descriptions object.
     *
     * @return The C enyx_hw_core_descriptions object.
     *
     * @since 5.0.0
     */
    enyx_hw_core_descriptions *
    handle() noexcept;

    /**
     * Direct access to the underlying C enyx_hw_core_descriptions object.
     *
     * @return The C enyx_hw_core_descriptions object.
     *
     * @since 5.0.0
     */
    enyx_hw_core_descriptions const *
    handle() const noexcept;

private:
    std::shared_ptr<enyx_hw_core_descriptions> descriptions_;
};


/**
 * Creates the Union of @p rhs and @p lhs.
 *
 * A failure can happen if both core_descriptions have the same hardware_id
 * for two cores with different names (unlikely).
 *
 * @return The Union of @p rhs and @p lhs.
 * @throws std::system_error if @p rhs is not consistent with @p lhs.
 *
 * @since 5.0.0
 */
core_descriptions
operator+(core_descriptions lhs, const core_descriptions & rhs);

/**
 * Build @b core_descriptions from a folder containg xml files or a single
 * xml file.
 *
 * @param path the path to the folder/file.
 * @return A result object containing either the new cores_descriptions or an
 *         errors.
 * Possible error codes are:
 *      - @b std::errc::no_such_file_or_directory if the path does not exists.
 *      - @b std::errc::invalid_argument if the path exists but is neither
 *           a regular file nor a directory.
 *      - @b std::errc::invalid_argument if the parsiing of xml files failed.
 *
 * @since 5.0.0
 */
result<core_descriptions>
core_descriptions_from_xml(std::string const& path) noexcept;

/**
 * Validate an xml file containing core descriptions using an XML Schema
 * Definition.
 *
 * @param xml_file The xml file to validate
 * @param xsd_file the schema to use
 * @return A result object containing an error on failure.
 *
 * @since 5.0.0
 */
result<void>
validate_xml_core_descriptions(std::string const & xml_file, std::string const & xsd_file) noexcept;

/**
 * Create a shared_ptr to a C core_descriptions from a non-NULL pointer.
 *
 * @param descriptions The descriptions to wrap
 * @return The wrapped descriptions (with destructor set).
 * @throws std::system_error if descriptions is nullptr.
 *
 * @since 5.0.0
 */
std::shared_ptr<enyx_hw_core_descriptions>
make_core_descriptions_ptr(enyx_hw_core_descriptions * descriptions);

ENYX_HW_NAMESPACE_END

#include <enyx/hw/core_descriptions.ipp>
